"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const asset_manifest_1 = require("./private/asset-manifest");
const toposort_1 = require("./private/toposort");
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 */
class CdkStage extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
        this.node.applyAspect({ visit: () => this.prepareStage() });
    }
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage, options = {}) {
        const asm = appStage.synth();
        const sortedTranches = toposort_1.topologicalSort(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(2); // We need 2 actions
            let executeRunOrder = runOrder + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: executeRunOrder });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        var _a, _b;
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(2);
        const executeRunOrder = (_b = options.executeRunOrder) !== null && _b !== void 0 ? _b : runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options = {}) {
        var _a;
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(),
        }));
    }
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepareStage() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder: runOrder, stackArtifact } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder: runOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = actions_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publishg the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = actions_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                });
            }
        }
    }
}
exports.CdkStage = CdkStage;
/**
 * A single output of a Stack
 */
class StackOutput {
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,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