"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Type of the asset that is being published
 */
var AssetType;
(function (AssetType) {
    /**
     * A file
     */
    AssetType["FILE"] = "file";
    /**
     * A Docker image
     */
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * Action to publish an asset in the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 */
class PublishAssetsAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.listValue({ produce: () => this.commands }),
                    },
                },
            }),
            // Needed to perform Docker builds
            environment: props.assetType === AssetType.DOCKER_IMAGE ? { privileged: true } : undefined,
            role: props.role,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
            role: props.role,
        });
    }
    /**
     * Add a single publishing command
     *
     * Manifest path should be relative to the root Cloud Assembly.
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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