"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node._actualNode;
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...node._aspects];
        const nodeAspectsCount = node._aspects.length;
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== node._aspects.length) {
                construct.node.addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
        };
        if (construct instanceof stack_1.Stack) {
            construct._synthesizeTemplate(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        // this will soon be deprecated and removed in 2.x
        // see https://github.com/aws/aws-cdk-rfcs/issues/192
        construct.onSynthesize(session);
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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