"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs'];
// list of NPM packages included in version reporting
const WHITELIST_PACKAGES = ['aws-rfdk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || WHITELIST_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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