"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs-extra");
const assets_1 = require("./assets");
const fs_1 = require("./fs");
const stage_1 = require("./stage");
const construct_compat_1 = require("./construct-compat");
const STAGING_TMP = '.cdk.staging';
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
        const outdir = (_c = stage_1.Stage.of(this)) === null || _c === void 0 ? void 0 : _c.outdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.stageAsset(outdir);
    }
    stageAsset(outdir) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Move bundling directory to staging directory
            fs.moveSync(this.bundleDir, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Temp staging directory in the working directory
        const stagingTmp = path.join('.', STAGING_TMP);
        fs.ensureDirSync(stagingTmp);
        // Create temp directory for bundling inside the temp staging directory
        const bundleDir = path.resolve(fs.mkdtempSync(path.join(stagingTmp, 'asset-bundle-')));
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            options.image._run({
                command: options.command,
                user,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : AssetStaging.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${AssetStaging.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                // Hash the hash to make sure we can use it in a file/directory name.
                // The resulting hash will also have the same length as for the other hash types.
                return crypto.createHash('sha256').update(props.assetHash).digest('hex');
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
//# sourceMappingURL=data:application/json;base64,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