"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        props.retryAttempts !== undefined && cdk.withResolved(props.retryAttempts, (attempts) => {
            if (attempts < 0 || attempts > 10000) {
                throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${attempts}`);
            }
        });
        props.parallelizationFactor !== undefined && cdk.withResolved(props.parallelizationFactor, (factor) => {
            if (factor < 1 || factor > 10) {
                throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${factor}`);
            }
        });
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB or Kinesis stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnQtc291cmNlLW1hcHBpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJldmVudC1zb3VyY2UtbWFwcGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxrQ0FBa0MsQ0FBQyxnREFBZ0Q7QUFHbkYseURBQTJEO0FBcUczRDs7Ozs7Ozs7Ozs7R0FXRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFXaEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE4Qjs7UUFDeEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLEdBQUcsR0FBRyxFQUFFO1lBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDL0c7UUFDRCxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO1lBQ25ILE1BQU0sSUFBSSxLQUFLLENBQUMsOERBQThELENBQUMsQ0FBQztTQUNuRjtRQUNELEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxDQUFDLFFBQVEsRUFBRSxFQUFFO1lBQ3BGLElBQUksUUFBUSxHQUFHLENBQUMsSUFBSSxRQUFRLEdBQUcsS0FBSyxFQUFFO2dCQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLDREQUE0RCxRQUFRLEVBQUUsQ0FBQyxDQUFDO2FBQzNGO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMscUJBQXFCLEtBQUssU0FBUyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLHFCQUFxQixFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUU7WUFDbEcsSUFBSSxNQUFNLEdBQUcsQ0FBQyxJQUFJLE1BQU0sR0FBRyxFQUFFLEVBQUU7Z0JBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLE1BQU0sRUFBRSxDQUFDLENBQUM7YUFDOUY7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUNILElBQUksaUJBQWlCLENBQUM7UUFDdEIsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ2pCLGlCQUFpQixHQUFHO2dCQUNoQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUM7YUFDdEQsQ0FBQztTQUNMO1FBQ0QsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLHdDQUFxQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDdEUsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLDBCQUEwQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7WUFDcEQsaUJBQWlCO1lBQ2pCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsWUFBWTtZQUN2QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLDhCQUE4QixRQUFFLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsU0FBUyxFQUFFO1lBQ3BFLHlCQUF5QixRQUFFLEtBQUssQ0FBQyxZQUFZLDBDQUFFLFNBQVMsRUFBRTtZQUMxRCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUN6QyxxQkFBcUIsRUFBRSxLQUFLLENBQUMscUJBQXFCO1NBQ3JELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxxQkFBcUIsQ0FBQyxHQUFHLENBQUM7SUFDMUQsQ0FBQztJQWhERDs7T0FFRztJQUNJLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSxvQkFBNEI7UUFDakcsTUFBTSxNQUFPLFNBQVEsR0FBRyxDQUFDLFFBQVE7WUFBakM7O2dCQUNvQix5QkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztZQUNoRSxDQUFDO1NBQUE7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNqQyxDQUFDO0NBeUNKO0FBbERELGdEQWtEQztBQUNEOzs7R0FHRztBQUNILElBQVksZ0JBV1g7QUFYRCxXQUFZLGdCQUFnQjtJQUN4Qjs7O09BR0c7SUFDSCxpREFBNkIsQ0FBQTtJQUM3Qjs7O09BR0c7SUFDSCxxQ0FBaUIsQ0FBQTtBQUNyQixDQUFDLEVBWFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFXM0IiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBJRXZlbnRTb3VyY2VEbHEgfSBmcm9tICcuL2RscSc7XG5pbXBvcnQgeyBJRnVuY3Rpb24gfSBmcm9tICcuL2Z1bmN0aW9uLWJhc2UnO1xuaW1wb3J0IHsgQ2ZuRXZlbnRTb3VyY2VNYXBwaW5nIH0gZnJvbSAnLi9sYW1iZGEuZ2VuZXJhdGVkJztcbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnRTb3VyY2VNYXBwaW5nT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIG9mIHRoZSBldmVudCBzb3VyY2UuIEFueSByZWNvcmQgYWRkZWQgdG9cbiAgICAgKiB0aGlzIHN0cmVhbSBjYW4gaW52b2tlIHRoZSBMYW1iZGEgZnVuY3Rpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgZXZlbnRTb3VyY2VBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbGFyZ2VzdCBudW1iZXIgb2YgcmVjb3JkcyB0aGF0IEFXUyBMYW1iZGEgd2lsbCByZXRyaWV2ZSBmcm9tIHlvdXIgZXZlbnRcbiAgICAgKiBzb3VyY2UgYXQgdGhlIHRpbWUgb2YgaW52b2tpbmcgeW91ciBmdW5jdGlvbi4gWW91ciBmdW5jdGlvbiByZWNlaXZlcyBhblxuICAgICAqIGV2ZW50IHdpdGggYWxsIHRoZSByZXRyaWV2ZWQgcmVjb3Jkcy5cbiAgICAgKlxuICAgICAqIFZhbGlkIFJhbmdlOiBNaW5pbXVtIHZhbHVlIG9mIDEuIE1heGltdW0gdmFsdWUgb2YgMTAwMDAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBLaW5lc2lzIGFuZCBBbWF6b24gRHluYW1vREIgaXMgMTAwIHJlY29yZHMuXG4gICAgICogQm90aCB0aGUgZGVmYXVsdCBhbmQgbWF4aW11bSBmb3IgQW1hem9uIFNRUyBhcmUgMTAgbWVzc2FnZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYmF0Y2hTaXplPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIElmIHRoZSBmdW5jdGlvbiByZXR1cm5zIGFuIGVycm9yLCBzcGxpdCB0aGUgYmF0Y2ggaW4gdHdvIGFuZCByZXRyeS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYmlzZWN0QmF0Y2hPbkVycm9yPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBBbiBBbWF6b24gU1FTIHF1ZXVlIG9yIEFtYXpvbiBTTlMgdG9waWMgZGVzdGluYXRpb24gZm9yIGRpc2NhcmRlZCByZWNvcmRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgZGlzY2FyZGVkIHJlY29yZHMgYXJlIGlnbm9yZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBvbkZhaWx1cmU/OiBJRXZlbnRTb3VyY2VEbHE7XG4gICAgLyoqXG4gICAgICogU2V0IHRvIGZhbHNlIHRvIGRpc2FibGUgdGhlIGV2ZW50IHNvdXJjZSB1cG9uIGNyZWF0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBwb3NpdGlvbiBpbiB0aGUgRHluYW1vREIgb3IgS2luZXNpcyBzdHJlYW0gd2hlcmUgQVdTIExhbWJkYSBzaG91bGRcbiAgICAgKiBzdGFydCByZWFkaW5nLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20va2luZXNpcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTaGFyZEl0ZXJhdG9yLmh0bWwjS2luZXNpcy1HZXRTaGFyZEl0ZXJhdG9yLXJlcXVlc3QtU2hhcmRJdGVyYXRvclR5cGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gUmVxdWlyZWQgZm9yIEFtYXpvbiBLaW5lc2lzIGFuZCBBbWF6b24gRHluYW1vREIgU3RyZWFtcyBzb3VyY2VzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YXJ0aW5nUG9zaXRpb24/OiBTdGFydGluZ1Bvc2l0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXhpbXVtIGFtb3VudCBvZiB0aW1lIHRvIGdhdGhlciByZWNvcmRzIGJlZm9yZSBpbnZva2luZyB0aGUgZnVuY3Rpb24uXG4gICAgICogTWF4aW11bSBvZiBEdXJhdGlvbi5taW51dGVzKDUpXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDApXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4QmF0Y2hpbmdXaW5kb3c/OiBjZGsuRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIG1heGltdW0gYWdlIG9mIGEgcmVjb3JkIHRoYXQgTGFtYmRhIHNlbmRzIHRvIGEgZnVuY3Rpb24gZm9yIHByb2Nlc3NpbmcuXG4gICAgICogVmFsaWQgUmFuZ2U6XG4gICAgICogKiBNaW5pbXVtIHZhbHVlIG9mIDYwIHNlY29uZHNcbiAgICAgKiAqIE1heGltdW0gdmFsdWUgb2YgNyBkYXlzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDcpXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4UmVjb3JkQWdlPzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiB0aW1lcyB0byByZXRyeSB3aGVuIHRoZSBmdW5jdGlvbiByZXR1cm5zIGFuIGVycm9yLlxuICAgICAqXG4gICAgICogVmFsaWQgUmFuZ2U6XG4gICAgICogKiBNaW5pbXVtIHZhbHVlIG9mIDBcbiAgICAgKiAqIE1heGltdW0gdmFsdWUgb2YgMTAwMDBcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDEwMDAwXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmV0cnlBdHRlbXB0cz86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIGJhdGNoZXMgdG8gcHJvY2VzcyBmcm9tIGVhY2ggc2hhcmQgY29uY3VycmVudGx5LlxuICAgICAqIFZhbGlkIFJhbmdlOlxuICAgICAqICogTWluaW11bSB2YWx1ZSBvZiAxXG4gICAgICogKiBNYXhpbXVtIHZhbHVlIG9mIDEwXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGFyYWxsZWxpemF0aW9uRmFjdG9yPzogbnVtYmVyO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBkZWNsYXJpbmcgYSBuZXcgZXZlbnQgc291cmNlIG1hcHBpbmcuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnRTb3VyY2VNYXBwaW5nUHJvcHMgZXh0ZW5kcyBFdmVudFNvdXJjZU1hcHBpbmdPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0IEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFyZ2V0OiBJRnVuY3Rpb247XG59XG4vKipcbiAqIFJlcHJlc2VudHMgYW4gZXZlbnQgc291cmNlIG1hcHBpbmcgZm9yIGEgbGFtYmRhIGZ1bmN0aW9uLlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGFtYmRhL2xhdGVzdC9kZy9pbnZvY2F0aW9uLWV2ZW50c291cmNlbWFwcGluZy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUV2ZW50U291cmNlTWFwcGluZyBleHRlbmRzIGNkay5JUmVzb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFRoZSBpZGVudGlmaWVyIGZvciB0aGlzIEV2ZW50U291cmNlTWFwcGluZ1xuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBldmVudFNvdXJjZU1hcHBpbmdJZDogc3RyaW5nO1xufVxuLyoqXG4gKiBEZWZpbmVzIGEgTGFtYmRhIEV2ZW50U291cmNlTWFwcGluZyByZXNvdXJjZS5cbiAqXG4gKiBVc3VhbGx5LCB5b3Ugd29uJ3QgbmVlZCB0byBkZWZpbmUgdGhlIG1hcHBpbmcgeW91cnNlbGYuIFRoaXMgd2lsbCB1c3VhbGx5IGJlIGRvbmUgYnlcbiAqIGV2ZW50IHNvdXJjZXMuIEZvciBleGFtcGxlLCB0byBhZGQgYW4gU1FTIGV2ZW50IHNvdXJjZSB0byBhIGZ1bmN0aW9uOlxuICpcbiAqICAgIGltcG9ydCB7IFNxc0V2ZW50U291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcbiAqICAgIGxhbWJkYS5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2Uoc3FzKSk7XG4gKlxuICogVGhlIGBTcXNFdmVudFNvdXJjZWAgY2xhc3Mgd2lsbCBhdXRvbWF0aWNhbGx5IGNyZWF0ZSB0aGUgbWFwcGluZywgYW5kIHdpbGwgYWxzb1xuICogbW9kaWZ5IHRoZSBMYW1iZGEncyBleGVjdXRpb24gcm9sZSBzbyBpdCBjYW4gY29uc3VtZSBtZXNzYWdlcyBmcm9tIHRoZSBxdWV1ZS5cbiAqL1xuZXhwb3J0IGNsYXNzIEV2ZW50U291cmNlTWFwcGluZyBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFdmVudFNvdXJjZU1hcHBpbmcge1xuICAgIC8qKlxuICAgICAqIEltcG9ydCBhbiBldmVudCBzb3VyY2UgaW50byB0aGlzIHN0YWNrIGZyb20gaXRzIGV2ZW50IHNvdXJjZSBpZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21FdmVudFNvdXJjZU1hcHBpbmdJZChzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgZXZlbnRTb3VyY2VNYXBwaW5nSWQ6IHN0cmluZyk6IElFdmVudFNvdXJjZU1hcHBpbmcge1xuICAgICAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJRXZlbnRTb3VyY2VNYXBwaW5nIHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBldmVudFNvdXJjZU1hcHBpbmdJZCA9IGV2ZW50U291cmNlTWFwcGluZ0lkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gICAgfVxuICAgIHB1YmxpYyByZWFkb25seSBldmVudFNvdXJjZU1hcHBpbmdJZDogc3RyaW5nO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRXZlbnRTb3VyY2VNYXBwaW5nUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgaWYgKHByb3BzLm1heEJhdGNoaW5nV2luZG93ICYmIHByb3BzLm1heEJhdGNoaW5nV2luZG93LnRvU2Vjb25kcygpID4gMzAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYG1heEJhdGNoaW5nV2luZG93IGNhbm5vdCBiZSBvdmVyIDMwMCBzZWNvbmRzLCBnb3QgJHtwcm9wcy5tYXhCYXRjaGluZ1dpbmRvdy50b1NlY29uZHMoKX1gKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMubWF4UmVjb3JkQWdlICYmIChwcm9wcy5tYXhSZWNvcmRBZ2UudG9TZWNvbmRzKCkgPCA2MCB8fCBwcm9wcy5tYXhSZWNvcmRBZ2UudG9EYXlzKHsgaW50ZWdyYWw6IGZhbHNlIH0pID4gNykpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignbWF4UmVjb3JkQWdlIG11c3QgYmUgYmV0d2VlbiA2MCBzZWNvbmRzIGFuZCA3IGRheXMgaW5jbHVzaXZlJyk7XG4gICAgICAgIH1cbiAgICAgICAgcHJvcHMucmV0cnlBdHRlbXB0cyAhPT0gdW5kZWZpbmVkICYmIGNkay53aXRoUmVzb2x2ZWQocHJvcHMucmV0cnlBdHRlbXB0cywgKGF0dGVtcHRzKSA9PiB7XG4gICAgICAgICAgICBpZiAoYXR0ZW1wdHMgPCAwIHx8IGF0dGVtcHRzID4gMTAwMDApIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHJldHJ5QXR0ZW1wdHMgbXVzdCBiZSBiZXR3ZWVuIDAgYW5kIDEwMDAwIGluY2x1c2l2ZSwgZ290ICR7YXR0ZW1wdHN9YCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgICBwcm9wcy5wYXJhbGxlbGl6YXRpb25GYWN0b3IgIT09IHVuZGVmaW5lZCAmJiBjZGsud2l0aFJlc29sdmVkKHByb3BzLnBhcmFsbGVsaXphdGlvbkZhY3RvciwgKGZhY3RvcikgPT4ge1xuICAgICAgICAgICAgaWYgKGZhY3RvciA8IDEgfHwgZmFjdG9yID4gMTApIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHBhcmFsbGVsaXphdGlvbkZhY3RvciBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMTAgaW5jbHVzaXZlLCBnb3QgJHtmYWN0b3J9YCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgICBsZXQgZGVzdGluYXRpb25Db25maWc7XG4gICAgICAgIGlmIChwcm9wcy5vbkZhaWx1cmUpIHtcbiAgICAgICAgICAgIGRlc3RpbmF0aW9uQ29uZmlnID0ge1xuICAgICAgICAgICAgICAgIG9uRmFpbHVyZTogcHJvcHMub25GYWlsdXJlLmJpbmQodGhpcywgcHJvcHMudGFyZ2V0KSxcbiAgICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgY2ZuRXZlbnRTb3VyY2VNYXBwaW5nID0gbmV3IENmbkV2ZW50U291cmNlTWFwcGluZyh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBiYXRjaFNpemU6IHByb3BzLmJhdGNoU2l6ZSxcbiAgICAgICAgICAgIGJpc2VjdEJhdGNoT25GdW5jdGlvbkVycm9yOiBwcm9wcy5iaXNlY3RCYXRjaE9uRXJyb3IsXG4gICAgICAgICAgICBkZXN0aW5hdGlvbkNvbmZpZyxcbiAgICAgICAgICAgIGVuYWJsZWQ6IHByb3BzLmVuYWJsZWQsXG4gICAgICAgICAgICBldmVudFNvdXJjZUFybjogcHJvcHMuZXZlbnRTb3VyY2VBcm4sXG4gICAgICAgICAgICBmdW5jdGlvbk5hbWU6IHByb3BzLnRhcmdldC5mdW5jdGlvbk5hbWUsXG4gICAgICAgICAgICBzdGFydGluZ1Bvc2l0aW9uOiBwcm9wcy5zdGFydGluZ1Bvc2l0aW9uLFxuICAgICAgICAgICAgbWF4aW11bUJhdGNoaW5nV2luZG93SW5TZWNvbmRzOiBwcm9wcy5tYXhCYXRjaGluZ1dpbmRvdz8udG9TZWNvbmRzKCksXG4gICAgICAgICAgICBtYXhpbXVtUmVjb3JkQWdlSW5TZWNvbmRzOiBwcm9wcy5tYXhSZWNvcmRBZ2U/LnRvU2Vjb25kcygpLFxuICAgICAgICAgICAgbWF4aW11bVJldHJ5QXR0ZW1wdHM6IHByb3BzLnJldHJ5QXR0ZW1wdHMsXG4gICAgICAgICAgICBwYXJhbGxlbGl6YXRpb25GYWN0b3I6IHByb3BzLnBhcmFsbGVsaXphdGlvbkZhY3RvcixcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuZXZlbnRTb3VyY2VNYXBwaW5nSWQgPSBjZm5FdmVudFNvdXJjZU1hcHBpbmcucmVmO1xuICAgIH1cbn1cbi8qKlxuICogVGhlIHBvc2l0aW9uIGluIHRoZSBEeW5hbW9EQiBvciBLaW5lc2lzIHN0cmVhbSB3aGVyZSBBV1MgTGFtYmRhIHNob3VsZCBzdGFydFxuICogcmVhZGluZy5cbiAqL1xuZXhwb3J0IGVudW0gU3RhcnRpbmdQb3NpdGlvbiB7XG4gICAgLyoqXG4gICAgICogU3RhcnQgcmVhZGluZyBhdCB0aGUgbGFzdCB1bnRyaW1tZWQgcmVjb3JkIGluIHRoZSBzaGFyZCBpbiB0aGUgc3lzdGVtLFxuICAgICAqIHdoaWNoIGlzIHRoZSBvbGRlc3QgZGF0YSByZWNvcmQgaW4gdGhlIHNoYXJkLlxuICAgICAqL1xuICAgIFRSSU1fSE9SSVpPTiA9ICdUUklNX0hPUklaT04nLFxuICAgIC8qKlxuICAgICAqIFN0YXJ0IHJlYWRpbmcganVzdCBhZnRlciB0aGUgbW9zdCByZWNlbnQgcmVjb3JkIGluIHRoZSBzaGFyZCwgc28gdGhhdCB5b3VcbiAgICAgKiBhbHdheXMgcmVhZCB0aGUgbW9zdCByZWNlbnQgZGF0YSBpbiB0aGUgc2hhcmRcbiAgICAgKi9cbiAgICBMQVRFU1QgPSAnTEFURVNUJ1xufVxuIl19