"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HelmChart = void 0;
const aws_cloudformation_1 = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const kubectl_layer_1 = require("./kubectl-layer");
/**
 * Represents a helm chart within the Kubernetes system.
 *
 * Applies/deletes the resources using `kubectl` in sync with the resource.
 */
class HelmChart extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = core_1.Stack.of(this);
        // we maintain a single manifest custom resource handler for each cluster
        const handler = this.getOrCreateHelmChartHandler(props.cluster);
        if (!handler) {
            throw new Error('Cannot define a Helm chart on a cluster with kubectl disabled');
        }
        new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: aws_cloudformation_1.CustomResourceProvider.lambda(handler),
            resourceType: HelmChart.RESOURCE_TYPE,
            properties: {
                Release: props.release || this.node.uniqueId.slice(-63).toLowerCase(),
                Chart: props.chart,
                Version: props.version,
                Values: (props.values ? stack.toJsonString(props.values) : undefined),
                Namespace: props.namespace || 'default',
                Repository: props.repository,
            },
        });
    }
    getOrCreateHelmChartHandler(cluster) {
        if (!cluster.kubectlEnabled) {
            return undefined;
        }
        let handler = cluster.node.tryFindChild('HelmChartHandler');
        if (!handler) {
            handler = new lambda.Function(cluster, 'HelmChartHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'helm-chart')),
                runtime: lambda.Runtime.PYTHON_3_7,
                handler: 'index.handler',
                timeout: core_1.Duration.minutes(15),
                layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this, { version: '2.0.0-beta1' })],
                memorySize: 256,
                environment: {
                    CLUSTER_NAME: cluster.clusterName,
                },
                // NOTE: we must use the default IAM role that's mapped to "system:masters"
                // as the execution role of this custom resource handler. This is the only
                // way to be able to interact with the cluster after it's been created.
                role: cluster._defaultMastersRole,
            });
        }
        return handler;
    }
}
exports.HelmChart = HelmChart;
/**
 * The CloudFormation reosurce type.
 */
HelmChart.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-HelmChart';
//# sourceMappingURL=data:application/json;base64,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