"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiHardwareType = exports.EcsOptimizedImage = exports.EcsOptimizedAmi = exports.WindowsOptimizedVersion = exports.Cluster = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_generated_1 = require("./ecs.generated");
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        const containerInsights = props.containerInsights !== undefined ? props.containerInsights : false;
        const clusterSettings = containerInsights ? [{ name: 'containerInsights', value: 'enabled' }] : undefined;
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImage || new EcsOptimizedAmi(),
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(autoScalingGroup, options);
        return autoScalingGroup;
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        // Tie instances to cluster
        autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
        if (!options.canContainersAccessInstanceRole) {
            // Deny containers access to instance metadata service
            // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
            autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
            autoScalingGroup.addUserData('sudo service iptables save');
            // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
            autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
        }
        if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
            autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        // But, scoped down to minimal permissions required.
        //  Notes:
        //   - 'ecs:CreateCluster' removed. The cluster already exists.
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DeregisterContainerInstance',
                'ecs:RegisterContainerInstance',
                'ecs:Submit*',
            ],
            resources: [
                this.clusterArn,
            ],
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These act on a cluster instance, and the instance doesn't exist until the service starts.
                // Thus, scope to the cluster using a condition.
                // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
                'ecs:Poll',
                'ecs:StartTelemetrySession',
            ],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': this.clusterArn },
            },
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These do not support resource constraints, and must be resource '*'
                'ecs:DiscoverPollEndpoint',
                'ecr:GetAuthorizationToken',
                // Preserved for backwards compatibility.
                // Users are able to enable cloudwatch agent using CDK. Existing
                // customers might be installing CW agent as part of user-data so if we
                // remove these permissions we will break that customer use cases.
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
            });
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        return this.metric('CPUReservation', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        return this.metric('MemoryReservation', props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
/**
 * ECS-optimized Windows version list
 */
var WindowsOptimizedVersion;
(function (WindowsOptimizedVersion) {
    WindowsOptimizedVersion["SERVER_2019"] = "2019";
    WindowsOptimizedVersion["SERVER_2016"] = "2016";
})(WindowsOptimizedVersion = exports.WindowsOptimizedVersion || (exports.WindowsOptimizedVersion = {}));
/*
 * TODO:v2.0.0 remove EcsOptimizedAmi
 */
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
class EcsOptimizedAmi {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = (props && props.hardwareType) || AmiHardwareType.STANDARD;
        if (props && props.generation) { // generation defined in the props object
            if (props.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX && this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Amazon Linux does not support special hardware type. Use Amazon Linux 2 instead');
            }
            else if (props.windowsVersion) {
                throw new Error('"windowsVersion" and Linux image "generation" cannot be both set');
            }
            else {
                this.generation = props.generation;
            }
        }
        else if (props && props.windowsVersion) {
            if (this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Windows Server does not support special hardware type');
            }
            else {
                this.windowsVersion = props.windowsVersion;
            }
        }
        else { // generation not defined in props object
            // always default to Amazon Linux v2 regardless of HW
            this.generation = ec2.AmazonLinuxGeneration.AMAZON_LINUX_2;
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedAmi = EcsOptimizedAmi;
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 */
class EcsOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = props && props.hardwareType;
        if (props.windowsVersion) {
            this.windowsVersion = props.windowsVersion;
        }
        else if (props.generation) {
            this.generation = props.generation;
        }
        else {
            throw new Error('This error should never be thrown');
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM
     *
     * @param hardwareType ECS-optimized AMI variant to use
     */
    static amazonLinux2(hardwareType = AmiHardwareType.STANDARD) {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2, hardwareType });
    }
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM
     */
    static amazonLinux() {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX });
    }
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM
     *
     * @param windowsVersion Windows Version to use
     */
    static windows(windowsVersion) {
        return new EcsOptimizedImage({ windowsVersion });
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedImage = EcsOptimizedImage;
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this.clusterArn = props.clusterArn !== undefined ? props.clusterArn : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
}
/**
 * The ECS-optimized AMI variant to use. For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 */
var AmiHardwareType;
(function (AmiHardwareType) {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     */
    AmiHardwareType["STANDARD"] = "Standard";
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     */
    AmiHardwareType["GPU"] = "GPU";
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     */
    AmiHardwareType["ARM"] = "ARM64";
})(AmiHardwareType = exports.AmiHardwareType || (exports.AmiHardwareType = {}));
//# sourceMappingURL=data:application/json;base64,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