"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return { originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
            } };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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