"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("../../aws-cloudfront"); // Automatically re-written from '@aws-cdk/aws-cloudfront'
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @experimental
 */
class S3Origin {
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, props) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(scope, 'S3Origin');
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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