"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupSelection = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
const backupable_resources_collector_1 = require("./backupable-resources-collector");
const resource_1 = require("./resource");
/**
 * A backup selection
 */
class BackupSelection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listOfTags = [];
        this.resources = [];
        this.backupableResourcesCollector = new backupable_resources_collector_1.BackupableResourcesCollector();
        const role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('backup.amazonaws.com'),
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForBackup'));
        if (props.allowRestores) {
            role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForRestores'));
        }
        this.grantPrincipal = role;
        const selection = new backup_generated_1.CfnBackupSelection(this, 'Resource', {
            backupPlanId: props.backupPlan.backupPlanId,
            backupSelection: {
                iamRoleArn: role.roleArn,
                selectionName: props.backupSelectionName || this.node.id,
                listOfTags: core_1.Lazy.anyValue({
                    produce: () => this.listOfTags,
                }, { omitEmptyArray: true }),
                resources: core_1.Lazy.listValue({
                    produce: () => [...this.resources, ...this.backupableResourcesCollector.resources],
                }, { omitEmpty: true }),
            },
        });
        this.backupPlanId = selection.attrBackupPlanId;
        this.selectionId = selection.attrSelectionId;
        for (const resource of props.resources) {
            this.addResource(resource);
        }
    }
    addResource(resource) {
        if (resource.tagCondition) {
            this.listOfTags.push({
                conditionKey: resource.tagCondition.key,
                conditionType: resource.tagCondition.operation || resource_1.TagOperation.STRING_EQUALS,
                conditionValue: resource.tagCondition.value,
            });
        }
        if (resource.resource) {
            this.resources.push(resource.resource);
        }
        if (resource.construct) {
            resource.construct.node.applyAspect(this.backupableResourcesCollector);
            // Cannot push `this.backupableResourcesCollector.resources` to
            // `this.resources` here because it has not been evaluated yet.
            // Will be concatenated to `this.resources` in a `Lazy.listValue`
            // in the constructor instead.
        }
    }
}
exports.BackupSelection = BackupSelection;
//# sourceMappingURL=data:application/json;base64,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