"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.extraLambdaEnv = {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props?.vpc,
            vpcSubnets: this.props?.vpcSubnets,
            allowPublicSubnet: this.props?.allowPublicSubnet,
            securityGroups: this.props?.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props?.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props?.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props?.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild'),
                new lambda_1.LambdaRunner(this, 'Lambda'),
                new fargate_1.FargateRunner(this, 'Fargate'),
            ];
        }
        this.checkIntersectingLabels();
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteRunnerFunction = this.deleteRunner();
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: false,
            }),
        });
        deleteRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const waitForIdleRunner = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, 'Wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.duration(props?.idleTimeout ?? cdk.Duration.minutes(10)),
        });
        const deleteIdleRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Idle Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: true,
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(...provider.labels.map(label => aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${label}`))), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .branch(waitForIdleRunner.next(deleteIdleRunnerTask))
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        let logOptions;
        if (this.props?.logOptions) {
            this.stateMachineLogGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                logGroupName: props?.logOptions?.logGroupName,
                retention: props?.logOptions?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            logOptions = {
                destination: this.stateMachineLogGroup,
                includeExecutionData: props?.logOptions?.includeExecutionData ?? true,
                level: props?.logOptions?.level ?? aws_cdk_lib_1.aws_stepfunctions.LogLevel.ALL,
            };
        }
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: work,
            logs: logOptions,
        });
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                STEP_FUNCTION_LOG_GROUP: this.stateMachineLogGroup?.logGroupName ?? '',
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
    checkIntersectingLabels() {
        // this "algorithm" is very inefficient, but good enough for the tiny datasets we expect
        for (const p1 of this.providers) {
            for (const p2 of this.providers) {
                if (p1 == p2) {
                    continue;
                }
                if (p1.labels.every(l => p2.labels.includes(l))) {
                    if (p2.labels.every(l => p1.labels.includes(l))) {
                        throw new Error(`Both ${p1.node.path} and ${p2.node.path} use the same labels [${p1.labels.join(', ')}]`);
                    }
                    aws_cdk_lib_1.Annotations.of(p1).addWarning(`Labels [${p1.labels.join(', ')}] intersect with another provider (${p2.node.path} -- [${p2.labels.join(', ')}]). If a workflow specifies the labels [${p1.labels.join(', ')}], it is not guaranteed which provider will be used. It is recommended you do not use intersecting labels`);
                }
            }
        }
    }
    /**
     * Metric for the number of GitHub Actions jobs completed. It has `ProviderLabels` and `Status` dimensions. The status can be one of "Succeeded", "SucceededWithIssues", "Failed", "Canceled", "Skipped", or "Abandoned".
     *
     * **WARNING:** this method creates a metric filter for each provider. Each metric has a status dimension with six possible values. These resources may incur cost.
     */
    metricJobCompleted(props) {
        if (!this.jobsCompletedMetricFilters) {
            // we can't use logs.FilterPattern.spaceDelimited() because it has no support for ||
            // status list taken from https://github.com/actions/runner/blob/be9632302ceef50bfb36ea998cea9c94c75e5d4d/src/Sdk/DTWebApi/WebApi/TaskResult.cs
            // we need "..." for Lambda that prefixes some extra data to log lines
            const pattern = aws_cdk_lib_1.aws_logs.FilterPattern.literal('[..., marker = "CDKGHA", job = "JOB", done = "DONE", labels, status = "Succeeded" || status = "SucceededWithIssues" || status = "Failed" || status = "Canceled" || status = "Skipped" || status = "Abandoned"]');
            this.jobsCompletedMetricFilters = this.providers.map(p => p.logGroup.addMetricFilter(`${p.logGroup.node.id} filter`, {
                metricNamespace: 'GitHubRunners',
                metricName: 'JobCompleted',
                filterPattern: pattern,
                metricValue: '1',
                // can't with dimensions -- defaultValue: 0,
                dimensions: {
                    ProviderLabels: '$labels',
                    Status: '$status',
                },
            }));
            for (const metricFilter of this.jobsCompletedMetricFilters) {
                if (metricFilter.node.defaultChild instanceof aws_cdk_lib_1.aws_logs.CfnMetricFilter) {
                    metricFilter.node.defaultChild.addPropertyOverride('MetricTransformations.0.Unit', 'Count');
                }
                else {
                    aws_cdk_lib_1.Annotations.of(metricFilter).addWarning('Unable to set metric filter Unit to Count');
                }
            }
        }
        return new aws_cdk_lib_1.aws_cloudwatch.Metric({
            namespace: 'GitHubRunners',
            metricName: 'JobsCompleted',
            unit: aws_cdk_lib_1.aws_cloudwatch.Unit.COUNT,
            statistic: aws_cdk_lib_1.aws_cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for successful executions.
     *
     * A successful execution doesn't always mean a runner was started. It can be successful even without any label matches.
     *
     * A successful runner doesn't mean the job it executed was successful. For that, see {@link metricJobCompleted}.
     */
    metricSucceeded(props) {
        return this.orchestrator.metricSucceeded(props);
    }
    /**
     * Metric for failed runner executions.
     *
     * A failed runner usually means the runner failed to start and so a job was never executed. It doesn't necessarily mean the job was executed and failed. For that, see {@link metricJobCompleted}.
     */
    metricFailed(props) {
        return this.orchestrator.metricFailed(props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes. This time may be longer than the time the runner took.
     */
    metricTime(props) {
        return this.orchestrator.metricTime(props);
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.8.0" };
//# sourceMappingURL=data:application/json;base64,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