"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
const utils_1 = require("../../utils");
const common_2 = require("../common");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 */
class AmiRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_1.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        let parentAmi;
        let workingDirectory;
        if (props.platform == 'Linux') {
            let archUrl;
            if (props.architecture.is(common_2.Architecture.X86_64)) {
                archUrl = 'amd64';
            }
            else if (props.architecture.is(common_2.Architecture.ARM64)) {
                archUrl = 'arm64';
            }
            else {
                throw new Error(`Unsupported architecture for parent AMI: ${props.architecture.name}`);
            }
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter(`/aws/service/canonical/ubuntu/server/focal/stable/current/${archUrl}/hvm/ebs-gp2/ami-id`, {
                os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
            }).getImage(this).imageId;
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.latestWindows(aws_cdk_lib_1.aws_ec2.WindowsVersion.WINDOWS_SERVER_2022_ENGLISH_FULL_CONTAINERSLATEST).getImage(this).imageId;
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ImageRecipe', name, {
                platform: props.platform,
                components,
                parentAmi,
            }),
            parentImage: parentAmi,
            components,
            workingDirectory,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An AMI builder that uses AWS Image Builder to build AMIs pre-baked with all the GitHub Actions runner requirements. Builders can be used with {@link Ec2Runner}.
 *
 * Each builder re-runs automatically at a set interval to make sure the AMIs contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the AMI used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the EC2 provider, use:
 *
 * ```
 * const builder = new AmiBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.addComponent(new ImageBuilderComponent(scope, id, {
 *   platform: 'Linux',
 *   displayName: 'p7zip',
 *   description: 'Install some more packages',
 *   commands: [
 *     'set -ex',
 *     'apt-get install p7zip',
 *   ],
 * }));
 * new Ec2Runner(this, 'EC2 provider', {
 *     label: 'custom-ec2',
 *     amiBuilder: builder,
 * });
 * ```
 */
class AmiBuilder extends common_1.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_2.Os.LINUX, common_2.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_2.Architecture.X86_64, common_2.Architecture.ARM64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'AMI',
        });
        // add all basic components
        if (this.os.is(common_2.Os.WINDOWS)) {
            this.addBaseWindowsComponents(props?.installDocker ?? true);
        }
        else if (this.os.is(common_2.Os.LINUX)) {
            this.addBaseLinuxComponents(props?.installDocker ?? true);
        }
    }
    addBaseWindowsComponents(installDocker) {
        this.addComponent(windows_components_1.WindowsComponents.cloudwatchAgent(this, 'CloudWatch agent'));
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
        if (installDocker) {
            this.addComponent(windows_components_1.WindowsComponents.docker(this, 'Docker'));
        }
    }
    addBaseLinuxComponents(installDocker) {
        this.addComponent(linux_components_1.LinuxUbuntuComponents.requiredPackages(this, 'Upgrade packages and install basics', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.runnerUser(this, 'User', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.awsCli(this, 'AWS CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubCli(this, 'GitHub CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.git(this, 'git', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion, this.architecture));
        if (installDocker) {
            this.addComponent(linux_components_1.LinuxUbuntuComponents.docker(this, 'Docker', this.architecture));
        }
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the AMI builder.
     */
    bind() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_1.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform,
            components: this.components,
            architecture: this.architecture,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        this.imageCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    imageCleaner(launchTemplate, stackName, builderName) {
        const deleter = utils_1.BundledNodejsFunction.singleton(this, 'delete-ami', {
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
}
exports.AmiBuilder = AmiBuilder;
_a = JSII_RTTI_SYMBOL_1;
AmiBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.AmiBuilder", version: "0.8.0" };
//# sourceMappingURL=data:application/json;base64,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