'''
Management command to set as None any CourseDailyMetrics.average_progress values
if produced by a backfill.

We do this because progress or any other value generated by examining StudentModule
values will not be correct if done for a previous date, until or unless Figures uses
StudentModuleHistory or Persistent Grades to examine db-stored student grade or SM values.
'''

from __future__ import absolute_import, print_function

from datetime import timedelta
from textwrap import dedent

from django.contrib.sites.models import Site
from django.db.models import Count, F

from figures.management.base import BaseBackfillCommand
from figures.models import CourseDailyMetrics


class Command(BaseBackfillCommand):
    '''Set all CourseDailyMetrics average_progress values to None where CDM was created
    more than one day after the date_for value.  See module docstring for rationale.
    '''

    help = dedent(__doc__).strip()

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run',
            action='store_true',
            default=False,
            help=('Dry run. Output but don\'t save changes')
        )
        super(Command, self).add_arguments(parser)

    def handle(self, *args, **options):
        '''
        '''
        site_id = self.get_site_ids(options['site'])[0]
        site = Site.objects.get(id=site_id)

        print('FIGURES: Repairing backfilled CDM.average_progress for site {}'.format(site))

        backfills = CourseDailyMetrics.objects.filter(
            site=site, created__gt=F('date_for') + timedelta(days=2),
            average_progress__isnull=False
        ).annotate(courses_count=Count('course_id', distinct=True))

        num_backfills = backfills.count()

        if num_backfills == 0:
            print('FIGURES: Found no CDM records with average_progress to repair.')
            return

        logmsg = (
            'FIGURES: Found {count} records from dates between {date_start} and {date_end} '
            'to update with None values for average_progress, from courses:\n\n{courses}.'
            '{dry_run_msg}'.format(
                count=num_backfills,
                date_start=backfills.earliest('date_for').date_for,
                date_end=backfills.latest('date_for').date_for,
                courses='\n'.join(set(backfills.values_list('course_id', flat=True))),
                dry_run_msg='\n\nDRY RUN.  Not updating records.' if options['dry_run'] else ''
            )
        )
        print(logmsg)

        if not options['dry_run']:
            print('FIGURES: set average_progress to None for {} CourseDailyMetrics records'.format(
                num_backfills
            ))
            backfills.update(average_progress=None)
