"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RestartPolicy = exports.Pod = exports.PodTemplate = exports.PodSpec = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const cdk8s_1 = require("cdk8s");
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 */
class PodSpec {
    constructor(props = {}) {
        var _a, _b;
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this._containers = (_a = props.containers) !== null && _a !== void 0 ? _a : [];
        this._volumes = (_b = props.volumes) !== null && _b !== void 0 ? _b : [];
    }
    get containers() {
        return [...this._containers];
    }
    get volumes() {
        return [...this._volumes];
    }
    addContainer(container) {
        this._containers.push(container);
    }
    addVolume(volume) {
        this._volumes.push(volume);
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _a;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = [];
        const containers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                volumes.push(mount.volume._toKube());
            }
            containers.push(container._toKube());
        }
        for (const volume of this._volumes) {
            volumes.push(volume._toKube());
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_a = this.serviceAccount) === null || _a === void 0 ? void 0 : _a.name,
            containers: containers,
            volumes: volumes,
        };
    }
}
exports.PodSpec = PodSpec;
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 */
class PodTemplate extends PodSpec {
    constructor(props = {}) {
        super(props);
        this.podMetadata = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadata);
    }
    /**
     * @internal
     */
    _toPodTemplateSpec() {
        return {
            metadata: this.podMetadata.toJson(),
            spec: this._toPodSpec(),
        };
    }
}
exports.PodTemplate = PodTemplate;
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, { metadata: props.metadata });
        this.apiObject = new k8s.Pod(this, 'Pod', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._spec._toPodSpec() }),
        });
        this._spec = new PodSpec(props);
    }
    get containers() {
        return this._spec.containers;
    }
    get volumes() {
        return this._spec.volumes;
    }
    get restartPolicy() {
        return this._spec.restartPolicy;
    }
    get serviceAccount() {
        return this._spec.serviceAccount;
    }
    addContainer(container) {
        return this._spec.addContainer(container);
    }
    addVolume(volume) {
        return this._spec.addVolume(volume);
    }
}
exports.Pod = Pod;
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
//# sourceMappingURL=data:application/json;base64,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