"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = void 0;
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
/**
 * A Job creates one or more Pods and ensures that a specified number of them successfully terminate. As pods successfully complete,
 * the Job tracks the successful completions. When a specified number of successful completions is reached, the task (ie, Job) is complete.
 * Deleting a Job will clean up the Pods it created. A simple case is to create one Job object in order to reliably run one Pod to completion.
 * The Job object will start a new Pod if the first Pod fails or is deleted (for example due to a node hardware failure or a node reboot).
 * You can also use a Job to run multiple Pods in parallel.
 */
class Job extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, { metadata: props.metadata });
        this.apiObject = new k8s.Job(this, 'Default', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this._podTemplate = new pod_1.PodTemplate({
            ...props,
            restartPolicy: (_a = props.restartPolicy) !== null && _a !== void 0 ? _a : pod_1.RestartPolicy.NEVER,
        });
        this.ttlAfterFinished = props.ttlAfterFinished;
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            template: this._podTemplate._toPodTemplateSpec(),
            ttlSecondsAfterFinished: this.ttlAfterFinished ? this.ttlAfterFinished.toSeconds() : undefined,
        };
    }
}
exports.Job = Job;
//# sourceMappingURL=data:application/json;base64,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