"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s_1 = require("cdk8s");
/**
 * Ingress is a collection of rules that allow inbound connections to reach the
 * endpoints defined by a backend. An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 */
class Ingress extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, { metadata: props.metadata });
        this._rulesPerHost = {};
        this.apiObject = new k8s.Ingress(this, 'Ingress', {
            metadata: props.metadata,
            spec: {
                backend: cdk8s_1.Lazy.any({ produce: () => { var _a; return (_a = this._defaultBackend) === null || _a === void 0 ? void 0 : _a._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_a = props.rules) !== null && _a !== void 0 ? _a : []);
    }
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress. A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name. This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match
     * @param backend The backend to route to
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific
     * HTTP path (the `Host` header matches this value).
     *
     * @param host The host name
     * @param path The HTTP path
     * @param backend The backend to route requests to
     */
    addHostRule(host, path, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path
     * @param backend The backend to route requests to
     */
    addRule(path, backend) {
        this.addRules({ backend, path });
    }
    /**
     * Adds rules to this ingress.
     * @param rules The rules to add
     */
    addRules(...rules) {
        var _a, _b;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_a = rule.host) !== null && _a !== void 0 ? _a : '';
            const backend = rule.backend;
            const path = rule.path;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_b = this._rulesPerHost[host]) !== null && _b !== void 0 ? _b : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({ backend: backend._toKube(), path });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
}
exports.Ingress = Ingress;
/**
 * The backend for an ingress path.
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     * @param service The service object.
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            serviceName: service.name,
            servicePort,
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
function sortByPath(lhs, rhs) {
    var _a, _b;
    const p1 = (_a = lhs.path) !== null && _a !== void 0 ? _a : '';
    const p2 = (_b = rhs.path) !== null && _b !== void 0 ? _b : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5ncmVzcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9pbmdyZXNzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFDQUFxQztBQUVyQyxpQ0FBaUQ7QUFDakQsaUNBQXdDO0FBNEJ4Qzs7Ozs7R0FLRztBQUNILE1BQWEsT0FBUSxTQUFRLGVBQVE7SUFVbkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFzQixFQUFFOztRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUpoQyxrQkFBYSxHQUE4QyxFQUFFLENBQUM7UUFNN0UsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNoRCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsSUFBSSxFQUFFO2dCQUNKLE9BQU8sRUFBRSxZQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSx3QkFBQyxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPLEtBQUUsRUFBRSxDQUFDO2dCQUNyRSxLQUFLLEVBQUUsWUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQzthQUN0RDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1NBQzlDO1FBRUQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFHLEtBQUssQ0FBQyxLQUFLLG1DQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFUyxVQUFVO1FBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDekUsT0FBTyxDQUFFLDBDQUEwQyxDQUFFLENBQUM7U0FDdkQ7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGlCQUFpQixDQUFDLE9BQXVCO1FBQzlDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUIsQ0FBQyxJQUFZLEVBQUUsT0FBdUI7UUFDaEUsSUFBSSxDQUFDLElBQUksRUFBRTtZQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLENBQUMsQ0FBQztTQUFFO1FBQ25FLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFdBQVcsQ0FBQyxJQUFZLEVBQUUsSUFBWSxFQUFFLE9BQXVCO1FBQ3BFLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7U0FBRTtRQUNuRSxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE9BQU8sQ0FBQyxJQUFZLEVBQUUsT0FBdUI7UUFDbEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7O09BR0c7SUFDSSxRQUFRLENBQUMsR0FBRyxLQUFvQjs7UUFDckMsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLEVBQUU7WUFFeEIsdUNBQXVDO1lBQ3ZDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtnQkFDNUIsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO29CQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLHVEQUF1RCxDQUFDLENBQUM7aUJBQzFFO2dCQUNELElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDcEMsU0FBUzthQUNWO1lBRUQsTUFBTSxJQUFJLFNBQUcsSUFBSSxDQUFDLElBQUksbUNBQUksRUFBRSxDQUFDO1lBQzdCLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUM7WUFDN0IsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztZQUV2QixJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLElBQUksRUFBRSxDQUFDLENBQUM7YUFDakU7WUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxTQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLG1DQUFJLEVBQUUsQ0FBQztZQUV6RSxxREFBcUQ7WUFDckQsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxJQUFJLENBQUMsRUFBRTtnQkFDckMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsSUFBSSxHQUFHLElBQUksRUFBRSxDQUFDLENBQUM7YUFDeEU7WUFFRCxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ25EO0lBQ0gsQ0FBQztJQUVPLFVBQVU7UUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQW1CLENBQUM7UUFFM0MsS0FBSyxNQUFNLENBQUUsSUFBSSxFQUFFLEtBQUssQ0FBRSxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ2hFLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ1QsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUM3QixJQUFJLEVBQUUsRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRTthQUN4QyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO0lBQzlDLENBQUM7Q0FDRjtBQS9IRCwwQkErSEM7QUFtQkQ7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFvQ3pCLFlBQXFDLE9BQTJCO1FBQTNCLFlBQU8sR0FBUCxPQUFPLENBQW9CO0lBRWhFLENBQUM7SUFyQ0Q7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFnQixFQUFFLFVBQXdDLEVBQUU7UUFDcEYsSUFBSSxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsSUFBSSxXQUFXLENBQUM7UUFDaEIsSUFBSSxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDOUIsV0FBVyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1NBQ3JDO2FBQU07WUFDTCxJQUFJLE9BQU8sQ0FBQyxJQUFJLEtBQUssU0FBUyxFQUFFO2dCQUM5QixNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUMvRCxJQUFJLEtBQUssRUFBRTtvQkFDVCxXQUFXLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztpQkFDMUI7cUJBQU07b0JBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7aUJBQ3pJO2FBQ0Y7aUJBQU07Z0JBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQzthQUN2STtTQUNGO1FBRUQsSUFBSSxPQUFPLENBQUMsSUFBSSxLQUFLLFNBQVMsSUFBSSxXQUFXLEtBQUssT0FBTyxDQUFDLElBQUksRUFBRTtZQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixPQUFPLENBQUMsSUFBSSw2QkFBNkIsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUNqRztRQUVELE9BQU8sSUFBSSxjQUFjLENBQUM7WUFDeEIsV0FBVyxFQUFFLE9BQU8sQ0FBQyxJQUFJO1lBQ3pCLFdBQVc7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDO0lBTUQ7O09BRUc7SUFDSSxPQUFPLEtBQUssT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztDQUMxQztBQTVDRCx3Q0E0Q0M7QUEwQ0QsU0FBUyxVQUFVLENBQUMsR0FBd0IsRUFBRSxHQUF3Qjs7SUFDcEUsTUFBTSxFQUFFLFNBQUcsR0FBRyxDQUFDLElBQUksbUNBQUksRUFBRSxDQUFDO0lBQzFCLE1BQU0sRUFBRSxTQUFHLEdBQUcsQ0FBQyxJQUFJLG1DQUFJLEVBQUUsQ0FBQztJQUMxQixPQUFPLEVBQUUsQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDOUIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgUmVzb3VyY2UsIFJlc291cmNlUHJvcHMgfSBmcm9tICcuL2Jhc2UnO1xuaW1wb3J0IHsgQXBpT2JqZWN0LCBMYXp5IH0gZnJvbSAnY2RrOHMnO1xuaW1wb3J0IHsgU2VydmljZSB9IGZyb20gJy4vc2VydmljZSc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYEluZ3Jlc3NgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluZ3Jlc3NQcm9wcyBleHRlbmRzIFJlc291cmNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgYmFja2VuZCBzZXJ2aWNlcyByZXF1ZXN0cyB0aGF0IGRvIG5vdCBtYXRjaCBhbnkgcnVsZS5cbiAgICpcbiAgICogVXNpbmcgdGhpcyBvcHRpb24gb3IgdGhlIGBhZGREZWZhdWx0QmFja2VuZCgpYCBtZXRob2QgaXMgZXF1aXZhbGVudCB0b1xuICAgKiBhZGRpbmcgYSBydWxlIHdpdGggYm90aCBgcGF0aGAgYW5kIGBob3N0YCB1bmRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0QmFja2VuZD86IEluZ3Jlc3NCYWNrZW5kO1xuXG4gIC8qKlxuICAgKiBSb3V0aW5nIHJ1bGVzIGZvciB0aGlzIGluZ3Jlc3MuXG4gICAqXG4gICAqIEVhY2ggcnVsZSBtdXN0IGRlZmluZSBhbiBgSW5ncmVzc0JhY2tlbmRgIHRoYXQgd2lsbCByZWNlaXZlIHRoZSByZXF1ZXN0c1xuICAgKiB0aGF0IG1hdGNoIHRoaXMgcnVsZS4gSWYgYm90aCBgaG9zdGAgYW5kIGBwYXRoYCBhcmUgbm90IHNwZWNpZmllYywgdGhpc1xuICAgKiBiYWNrZW5kIHdpbGwgYmUgdXNlZCBhcyB0aGUgZGVmYXVsdCBiYWNrZW5kIG9mIHRoZSBpbmdyZXNzLlxuICAgKlxuICAgKiBZb3UgY2FuIGFsc28gYWRkIHJ1bGVzIGxhdGVyIHVzaW5nIGBhZGRSdWxlKClgLCBgYWRkSG9zdFJ1bGUoKWAsXG4gICAqIGBhZGREZWZhdWx0QmFja2VuZCgpYCBhbmQgYGFkZEhvc3REZWZhdWx0QmFja2VuZCgpYC5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogSW5ncmVzc1J1bGVbXTtcbn1cblxuLyoqXG4gKiBJbmdyZXNzIGlzIGEgY29sbGVjdGlvbiBvZiBydWxlcyB0aGF0IGFsbG93IGluYm91bmQgY29ubmVjdGlvbnMgdG8gcmVhY2ggdGhlXG4gKiBlbmRwb2ludHMgZGVmaW5lZCBieSBhIGJhY2tlbmQuIEFuIEluZ3Jlc3MgY2FuIGJlIGNvbmZpZ3VyZWQgdG8gZ2l2ZSBzZXJ2aWNlc1xuICogZXh0ZXJuYWxseS1yZWFjaGFibGUgdXJscywgbG9hZCBiYWxhbmNlIHRyYWZmaWMsIHRlcm1pbmF0ZSBTU0wsIG9mZmVyIG5hbWVcbiAqIGJhc2VkIHZpcnR1YWwgaG9zdGluZyBldGMuXG4gKi9cbmV4cG9ydCBjbGFzcyBJbmdyZXNzIGV4dGVuZHMgUmVzb3VyY2Uge1xuXG4gIC8qKlxuICAgKiBAc2VlIGJhc2UuUmVzb3VyY2UuYXBpT2JqZWN0XG4gICAqL1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgYXBpT2JqZWN0OiBBcGlPYmplY3Q7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfcnVsZXNQZXJIb3N0OiB7IFtob3N0OiBzdHJpbmddOiBrOHMuSHR0cEluZ3Jlc3NQYXRoW10gfSA9IHt9O1xuICBwcml2YXRlIF9kZWZhdWx0QmFja2VuZD86IEluZ3Jlc3NCYWNrZW5kO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBJbmdyZXNzUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgeyBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEgfSk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuSW5ncmVzcyh0aGlzLCAnSW5ncmVzcycsIHtcbiAgICAgIG1ldGFkYXRhOiBwcm9wcy5tZXRhZGF0YSxcbiAgICAgIHNwZWM6IHtcbiAgICAgICAgYmFja2VuZDogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl9kZWZhdWx0QmFja2VuZD8uX3RvS3ViZSgpIH0pLFxuICAgICAgICBydWxlczogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnN5bnRoUnVsZXMoKSB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuZGVmYXVsdEJhY2tlbmQpIHtcbiAgICAgIHRoaXMuYWRkRGVmYXVsdEJhY2tlbmQocHJvcHMuZGVmYXVsdEJhY2tlbmQpO1xuICAgIH1cblxuICAgIHRoaXMuYWRkUnVsZXMoLi4ucHJvcHMucnVsZXMgPz8gW10pO1xuICB9XG5cbiAgcHJvdGVjdGVkIG9uVmFsaWRhdGUoKSB7XG4gICAgaWYgKCF0aGlzLl9kZWZhdWx0QmFja2VuZCAmJiBPYmplY3Qua2V5cyh0aGlzLl9ydWxlc1Blckhvc3QpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIFsgJ2luZ3Jlc3Mgd2l0aCBubyBydWxlcyBvciBkZWZhdWx0IGJhY2tlbmQnIF07XG4gICAgfVxuICAgIHJldHVybiBbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHRoZSBkZWZhdWx0IGJhY2tlbmQgZm9yIHRoaXMgaW5ncmVzcy4gQSBkZWZhdWx0IGJhY2tlbmQgY2FwYWJsZSBvZlxuICAgKiBzZXJ2aWNpbmcgcmVxdWVzdHMgdGhhdCBkb24ndCBtYXRjaCBhbnkgcnVsZS5cbiAgICpcbiAgICogQHBhcmFtIGJhY2tlbmQgVGhlIGJhY2tlbmQgdG8gdXNlIGZvciByZXF1ZXN0cyB0aGF0IGRvIG5vdCBtYXRjaCBhbnkgcnVsZS5cbiAgICovXG4gIHB1YmxpYyBhZGREZWZhdWx0QmFja2VuZChiYWNrZW5kOiBJbmdyZXNzQmFja2VuZCkge1xuICAgIHRoaXMuYWRkUnVsZXMoeyBiYWNrZW5kIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgYSBkZWZhdWx0IGJhY2tlbmQgZm9yIGEgc3BlY2lmaWMgaG9zdCBuYW1lLiBUaGlzIGJhY2tlbmQgd2lsbCBiZSB1c2VkIGFzIGEgY2F0Y2gtYWxsIGZvciByZXF1ZXN0c1xuICAgKiB0YXJnZXRlZCB0byB0aGlzIGhvc3QgbmFtZSAodGhlIGBIb3N0YCBoZWFkZXIgbWF0Y2hlcyB0aGlzIHZhbHVlKS5cbiAgICpcbiAgICogQHBhcmFtIGhvc3QgVGhlIGhvc3QgbmFtZSB0byBtYXRjaFxuICAgKiBAcGFyYW0gYmFja2VuZCBUaGUgYmFja2VuZCB0byByb3V0ZSB0b1xuICAgKi9cbiAgcHVibGljIGFkZEhvc3REZWZhdWx0QmFja2VuZChob3N0OiBzdHJpbmcsIGJhY2tlbmQ6IEluZ3Jlc3NCYWNrZW5kKSB7XG4gICAgaWYgKCFob3N0KSB7IHRocm93IG5ldyBFcnJvcignaG9zdCBtdXN0IG5vdCBiZSBhbiBlbXB0eSBzdHJpbmcnKTsgfVxuICAgIHRoaXMuYWRkUnVsZXMoeyBob3N0LCBiYWNrZW5kIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gaW5ncmVzcyBydWxlIGFwcGxpZWQgdG8gcmVxdWVzdHMgdG8gYSBzcGVjaWZpYyBob3N0IGFuZCBhIHNwZWNpZmljXG4gICAqIEhUVFAgcGF0aCAodGhlIGBIb3N0YCBoZWFkZXIgbWF0Y2hlcyB0aGlzIHZhbHVlKS5cbiAgICpcbiAgICogQHBhcmFtIGhvc3QgVGhlIGhvc3QgbmFtZVxuICAgKiBAcGFyYW0gcGF0aCBUaGUgSFRUUCBwYXRoXG4gICAqIEBwYXJhbSBiYWNrZW5kIFRoZSBiYWNrZW5kIHRvIHJvdXRlIHJlcXVlc3RzIHRvXG4gICAqL1xuICBwdWJsaWMgYWRkSG9zdFJ1bGUoaG9zdDogc3RyaW5nLCBwYXRoOiBzdHJpbmcsIGJhY2tlbmQ6IEluZ3Jlc3NCYWNrZW5kKSB7XG4gICAgaWYgKCFob3N0KSB7IHRocm93IG5ldyBFcnJvcignaG9zdCBtdXN0IG5vdCBiZSBhbiBlbXB0eSBzdHJpbmcnKTsgfVxuICAgIHRoaXMuYWRkUnVsZXMoeyBob3N0LCBiYWNrZW5kLCBwYXRoIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gaW5ncmVzcyBydWxlIGFwcGxpZWQgdG8gcmVxdWVzdHMgc2VudCB0byBhIHNwZWNpZmljIEhUVFAgcGF0aC5cbiAgICpcbiAgICogQHBhcmFtIHBhdGggVGhlIEhUVFAgcGF0aFxuICAgKiBAcGFyYW0gYmFja2VuZCBUaGUgYmFja2VuZCB0byByb3V0ZSByZXF1ZXN0cyB0b1xuICAgKi9cbiAgcHVibGljIGFkZFJ1bGUocGF0aDogc3RyaW5nLCBiYWNrZW5kOiBJbmdyZXNzQmFja2VuZCkge1xuICAgIHRoaXMuYWRkUnVsZXMoeyBiYWNrZW5kLCBwYXRoIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgcnVsZXMgdG8gdGhpcyBpbmdyZXNzLlxuICAgKiBAcGFyYW0gcnVsZXMgVGhlIHJ1bGVzIHRvIGFkZFxuICAgKi9cbiAgcHVibGljIGFkZFJ1bGVzKC4uLnJ1bGVzOiBJbmdyZXNzUnVsZVtdKSB7XG4gICAgZm9yIChjb25zdCBydWxlIG9mIHJ1bGVzKSB7XG5cbiAgICAgIC8vIGRlZmF1bHQgYmFja2VuZCBpcyBub3QgcmVhbGx5IGEgcnVsZVxuICAgICAgaWYgKCFydWxlLmhvc3QgJiYgIXJ1bGUucGF0aCkge1xuICAgICAgICBpZiAodGhpcy5fZGVmYXVsdEJhY2tlbmQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2EgZGVmYXVsdCBiYWNrZW5kIGlzIGFscmVhZHkgZGVmaW5lZCBmb3IgdGhpcyBpbmdyZXNzJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5fZGVmYXVsdEJhY2tlbmQgPSBydWxlLmJhY2tlbmQ7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBob3N0ID0gcnVsZS5ob3N0ID8/ICcnO1xuICAgICAgY29uc3QgYmFja2VuZCA9IHJ1bGUuYmFja2VuZDtcbiAgICAgIGNvbnN0IHBhdGggPSBydWxlLnBhdGg7XG5cbiAgICAgIGlmIChwYXRoICYmICFwYXRoLnN0YXJ0c1dpdGgoJy8nKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGluZ3Jlc3MgcGF0aHMgbXVzdCBiZWdpbiB3aXRoIGEgXCIvXCI6ICR7cGF0aH1gKTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgcm91dGVzID0gdGhpcy5fcnVsZXNQZXJIb3N0W2hvc3RdID0gdGhpcy5fcnVsZXNQZXJIb3N0W2hvc3RdID8/IFtdO1xuXG4gICAgICAvLyBjaGVjayBpZiB3ZSBhbHJlYWR5IGhhdmUgYSBydWxlIGZvciB0aGlzIGhvc3QvcGF0aFxuICAgICAgaWYgKHJvdXRlcy5maW5kKHIgPT4gci5wYXRoID09PSBwYXRoKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHRoZXJlIGlzIGFscmVhZHkgYW4gaW5ncmVzcyBydWxlIGZvciAke2hvc3R9JHtwYXRofWApO1xuICAgICAgfVxuXG4gICAgICByb3V0ZXMucHVzaCh7IGJhY2tlbmQ6IGJhY2tlbmQuX3RvS3ViZSgpLCBwYXRoIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc3ludGhSdWxlcygpOiB1bmRlZmluZWQgfCBrOHMuSW5ncmVzc1J1bGVbXSB7XG4gICAgY29uc3QgcnVsZXMgPSBuZXcgQXJyYXk8azhzLkluZ3Jlc3NSdWxlPigpO1xuXG4gICAgZm9yIChjb25zdCBbIGhvc3QsIHBhdGhzIF0gb2YgT2JqZWN0LmVudHJpZXModGhpcy5fcnVsZXNQZXJIb3N0KSkge1xuICAgICAgcnVsZXMucHVzaCh7XG4gICAgICAgIGhvc3Q6IGhvc3QgPyBob3N0IDogdW5kZWZpbmVkLFxuICAgICAgICBodHRwOiB7IHBhdGhzOiBwYXRocy5zb3J0KHNvcnRCeVBhdGgpIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gcnVsZXMubGVuZ3RoID4gMCA/IHJ1bGVzIDogdW5kZWZpbmVkO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3Igc2V0dGluZyB1cCBiYWNrZW5kcyBmb3IgaW5ncmVzcyBydWxlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2aWNlSW5ncmVzc0JhY2tlbmRPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwb3J0IHRvIHVzZSB0byBhY2Nlc3MgdGhlIHNlcnZpY2UuXG4gICAqXG4gICAqIC0gVGhpcyBvcHRpb24gd2lsbCBmYWlsIGlmIHRoZSBzZXJ2aWNlIGRvZXMgbm90IGV4cG9zZSBhbnkgcG9ydHMuXG4gICAqIC0gSWYgdGhlIHNlcnZpY2UgZXhwb3NlcyBtdWx0aXBsZSBwb3J0cywgdGhpcyBvcHRpb24gbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIC0gSWYgdGhlIHNlcnZpY2UgZXhwb3NlcyBhIHNpbmdsZSBwb3J0LCB0aGlzIG9wdGlvbiBpcyBvcHRpb25hbCBhbmQgaWZcbiAgICogICBzcGVjaWZpZWQsIGl0IG11c3QgYmUgdGhlIHNhbWUgcG9ydCBleHBvc2VkIGJ5IHRoZSBzZXJ2aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGlmIHRoZSBzZXJ2aWNlIGV4cG9zZXMgYSBzaW5nbGUgcG9ydCwgdGhpcyBwb3J0IHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnQ/OiBudW1iZXI7XG59XG5cbi8qKlxuICogVGhlIGJhY2tlbmQgZm9yIGFuIGluZ3Jlc3MgcGF0aC5cbiAqL1xuZXhwb3J0IGNsYXNzIEluZ3Jlc3NCYWNrZW5kIHtcbiAgLyoqXG4gICAqIEEgS3ViZXJuZXRlcyBgU2VydmljZWAgdG8gdXNlIGFzIHRoZSBiYWNrZW5kIGZvciB0aGlzIHBhdGguXG4gICAqIEBwYXJhbSBzZXJ2aWNlIFRoZSBzZXJ2aWNlIG9iamVjdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNlcnZpY2Uoc2VydmljZTogU2VydmljZSwgb3B0aW9uczogU2VydmljZUluZ3Jlc3NCYWNrZW5kT3B0aW9ucyA9IHt9KSB7XG4gICAgaWYgKHNlcnZpY2UucG9ydHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3NlcnZpY2UgZG9lcyBub3QgZXhwb3NlIGFueSBwb3J0cycpO1xuICAgIH1cblxuICAgIGxldCBzZXJ2aWNlUG9ydDtcbiAgICBpZiAoc2VydmljZS5wb3J0cy5sZW5ndGggPT09IDEpIHtcbiAgICAgIHNlcnZpY2VQb3J0ID0gc2VydmljZS5wb3J0c1swXS5wb3J0O1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAob3B0aW9ucy5wb3J0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgY29uc3QgZm91bmQgPSBzZXJ2aWNlLnBvcnRzLmZpbmQocCA9PiBwLnBvcnQgPT09IG9wdGlvbnMucG9ydCk7XG4gICAgICAgIGlmIChmb3VuZCkge1xuICAgICAgICAgIHNlcnZpY2VQb3J0ID0gZm91bmQucG9ydDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHNlcnZpY2UgZXhwb3NlcyBwb3J0cyAke3NlcnZpY2UucG9ydHMubWFwKHAgPT4gcC5wb3J0KS5qb2luKCcsJyl9IGJ1dCBiYWNrZW5kIGlzIGRlZmluZWQgdG8gdXNlIHBvcnQgJHtvcHRpb25zLnBvcnR9YCk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIGRldGVybWluZSBzZXJ2aWNlIHBvcnQgc2luY2Ugc2VydmljZSBleHBvc2VzIG11bHRpcGxlIHBvcnRzOiAke3NlcnZpY2UucG9ydHMubWFwKHggPT4geC5wb3J0KS5qb2luKCcsJyl9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucG9ydCAhPT0gdW5kZWZpbmVkICYmIHNlcnZpY2VQb3J0ICE9PSBvcHRpb25zLnBvcnQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgYmFja2VuZCBkZWZpbmVzIHBvcnQgJHtvcHRpb25zLnBvcnR9IGJ1dCBzZXJ2aWNlIGV4cG9zZXMgcG9ydCAke3NlcnZpY2VQb3J0fWApO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW5ncmVzc0JhY2tlbmQoe1xuICAgICAgc2VydmljZU5hbWU6IHNlcnZpY2UubmFtZSxcbiAgICAgIHNlcnZpY2VQb3J0LFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGJhY2tlbmQ6IGs4cy5JbmdyZXNzQmFja2VuZCkge1xuXG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpIHsgcmV0dXJuIHRoaXMuYmFja2VuZDsgfVxufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgdGhlIHJ1bGVzIG1hcHBpbmcgdGhlIHBhdGhzIHVuZGVyIGEgc3BlY2lmaWVkIGhvc3QgdG8gdGhlIHJlbGF0ZWRcbiAqIGJhY2tlbmQgc2VydmljZXMuIEluY29taW5nIHJlcXVlc3RzIGFyZSBmaXJzdCBldmFsdWF0ZWQgZm9yIGEgaG9zdCBtYXRjaCxcbiAqIHRoZW4gcm91dGVkIHRvIHRoZSBiYWNrZW5kIGFzc29jaWF0ZWQgd2l0aCB0aGUgbWF0Y2hpbmcgcGF0aC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbmdyZXNzUnVsZSB7XG4gIC8qKlxuICAgKiBCYWNrZW5kIGRlZmluZXMgdGhlIHJlZmVyZW5jZWQgc2VydmljZSBlbmRwb2ludCB0byB3aGljaCB0aGUgdHJhZmZpYyB3aWxsXG4gICAqIGJlIGZvcndhcmRlZCB0by5cbiAgICovXG4gIHJlYWRvbmx5IGJhY2tlbmQ6IEluZ3Jlc3NCYWNrZW5kO1xuXG4gIC8qKlxuICAgKiBIb3N0IGlzIHRoZSBmdWxseSBxdWFsaWZpZWQgZG9tYWluIG5hbWUgb2YgYSBuZXR3b3JrIGhvc3QsIGFzIGRlZmluZWQgYnlcbiAgICogUkZDIDM5ODYuIE5vdGUgdGhlIGZvbGxvd2luZyBkZXZpYXRpb25zIGZyb20gdGhlIFwiaG9zdFwiIHBhcnQgb2YgdGhlIFVSSSBhc1xuICAgKiBkZWZpbmVkIGluIHRoZSBSRkM6IDEuIElQcyBhcmUgbm90IGFsbG93ZWQuIEN1cnJlbnRseSBhbiBJbmdyZXNzUnVsZVZhbHVlXG4gICAqIGNhbiBvbmx5IGFwcGx5IHRvIHRoZSBJUCBpbiB0aGUgU3BlYyBvZiB0aGUgcGFyZW50IEluZ3Jlc3MuIDIuIFRoZSBgOmBcbiAgICogZGVsaW1pdGVyIGlzIG5vdCByZXNwZWN0ZWQgYmVjYXVzZSBwb3J0cyBhcmUgbm90IGFsbG93ZWQuIEN1cnJlbnRseSB0aGVcbiAgICogcG9ydCBvZiBhbiBJbmdyZXNzIGlzIGltcGxpY2l0bHkgOjgwIGZvciBodHRwIGFuZCA6NDQzIGZvciBodHRwcy4gQm90aFxuICAgKiB0aGVzZSBtYXkgY2hhbmdlIGluIHRoZSBmdXR1cmUuIEluY29taW5nIHJlcXVlc3RzIGFyZSBtYXRjaGVkIGFnYWluc3QgdGhlXG4gICAqIGhvc3QgYmVmb3JlIHRoZSBJbmdyZXNzUnVsZVZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElmIHRoZSBob3N0IGlzIHVuc3BlY2lmaWVkLCB0aGUgSW5ncmVzcyByb3V0ZXMgYWxsIHRyYWZmaWMgYmFzZWRcbiAgICogb24gdGhlIHNwZWNpZmllZCBJbmdyZXNzUnVsZVZhbHVlLlxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdD86IHN0cmluZztcblxuICAvKipcbiAgICogUGF0aCBpcyBhbiBleHRlbmRlZCBQT1NJWCByZWdleCBhcyBkZWZpbmVkIGJ5IElFRUUgU3RkIDEwMDMuMSwgKGkuZSB0aGlzXG4gICAqIGZvbGxvd3MgdGhlIGVncmVwL3VuaXggc3ludGF4LCBub3QgdGhlIHBlcmwgc3ludGF4KSBtYXRjaGVkIGFnYWluc3QgdGhlXG4gICAqIHBhdGggb2YgYW4gaW5jb21pbmcgcmVxdWVzdC4gQ3VycmVudGx5IGl0IGNhbiBjb250YWluIGNoYXJhY3RlcnMgZGlzYWxsb3dlZFxuICAgKiBmcm9tIHRoZSBjb252ZW50aW9uYWwgXCJwYXRoXCIgcGFydCBvZiBhIFVSTCBhcyBkZWZpbmVkIGJ5IFJGQyAzOTg2LiBQYXRoc1xuICAgKiBtdXN0IGJlZ2luIHdpdGggYSAnLycuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgdW5zcGVjaWZpZWQsIHRoZSBwYXRoIGRlZmF1bHRzIHRvIGEgY2F0Y2ggYWxsIHNlbmRpbmcgdHJhZmZpY1xuICAgKiB0byB0aGUgYmFja2VuZC5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGg/OiBzdHJpbmc7XG59XG5cbmZ1bmN0aW9uIHNvcnRCeVBhdGgobGhzOiBrOHMuSHR0cEluZ3Jlc3NQYXRoLCByaHM6IGs4cy5IdHRwSW5ncmVzc1BhdGgpIHtcbiAgY29uc3QgcDEgPSBsaHMucGF0aCA/PyAnJztcbiAgY29uc3QgcDIgPSByaHMucGF0aCA/PyAnJztcbiAgcmV0dXJuIHAxLmxvY2FsZUNvbXBhcmUocDIpO1xufVxuIl19