"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const k8s = require("./imports/k8s");
const constructs_1 = require("constructs");
const service_1 = require("./service");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const cdk8s_1 = require("cdk8s");
const pod_1 = require("./pod");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, { metadata: props.metadata });
        this.apiObject = new k8s.Deployment(this, 'Deployment', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.replicas = (_a = props.replicas) !== null && _a !== void 0 ? _a : 1;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_b = props.defaultSelector) !== null && _b !== void 0 ? _b : true) {
            const selector = 'cdk8s.deployment';
            const matcher = cdk8s_1.Names.toLabelValue(constructs_1.Node.of(this).path);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param port The port number the service will bind to.
     * @param options Options.
     */
    expose(port, options = {}) {
        var _a;
        const service = new service_1.Service(this, 'Service', {
            type: (_a = options.serviceType) !== null && _a !== void 0 ? _a : service_1.ServiceType.CLUSTER_IP,
        });
        service.addDeployment(this, port);
        return service;
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
        };
    }
}
exports.Deployment = Deployment;
//# sourceMappingURL=data:application/json;base64,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