import pandas as pd
from .encryption import HomomorphicEncryption

class SafePandas(pd.DataFrame):
    def __init__(self, *args, **kwargs):
        # Retrieve the key from kwargs, if provided
        key = kwargs.pop('key', None)
        super().__init__(*args, **kwargs)
        
        if key:
            self.encryption = HomomorphicEncryption(key)  # Use the provided key for encryption
        else:
            self.encryption = HomomorphicEncryption()  # If no key, generate a default key
        
        self.encrypted = True  # Flag to indicate data is encrypted
        self._encrypt_data()

    def _encrypt_data(self):
        """Encrypt all values in the DataFrame using Homomorphic Encryption"""
        for column in self.columns:
            self[column] = self[column].apply(lambda x: self.encryption.encrypt(x))

    def decrypt(self):
        """Decrypt the entire DataFrame using the encryption key."""
        decrypted_data = {column: self[column].apply(lambda x: self.encryption.decrypt(x)) for column in self.columns}
        decrypted_df = pd.DataFrame(decrypted_data)
        decrypted_df.encrypted = False
        return decrypted_df

    def to_csv(self, file_path, format="csv"):
        """Save the encrypted DataFrame to a file (CSV, Pickle)"""
        if format == "csv":
            super().to_csv(file_path, index=False)
        elif format == "pickle":
            super().to_pickle(file_path)
        else:
            raise ValueError("Unsupported format. Use 'csv' or 'pickle'.")
        
        # Save the key after saving the encrypted DataFrame
        self.encryption.save_key("encryption_key")

    @classmethod
    def read_csv(cls, file_path, format="csv", key=None):
        """Load the encrypted DataFrame from a file and return a SafePandas object."""
        if format == "csv":
            encrypted_df = pd.read_csv(file_path)
        elif format == "pickle":
            encrypted_df = pd.read_pickle(file_path)
        else:
            raise ValueError("Unsupported format. Use 'csv' or 'pickle'.")
        
        # Ensure the key is provided for decryption
        if not key:
            raise ValueError("A decryption key is required for loading encrypted data.")
        
        # Create a SafePandas object from the loaded encrypted DataFrame with the provided key
        return cls(encrypted_df, key=key)
