"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Spacer = exports.Column = exports.Row = void 0;
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row.
 *
 * Widgets will be laid out next to each other
 */
class Row {
    /**
     *
     */
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    /**
     * Place the widget at a given position.
     */
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column.
 *
 * Widgets will be laid out next to each other
 */
class Column {
    /**
     *
     */
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    /**
     * Place the widget at a given position.
     */
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space.
 */
class Spacer {
    /**
     *
     */
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    /**
     * Place the widget at a given position.
     */
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    /**
     * Return the widget JSON for use in the dashboard.
     */
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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