"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms.
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * Specify how missing data points are treated during alarm evaluation.
 */
var TreatMissingData;
(function (TreatMissingData) {
    TreatMissingData["BREACHING"] = "breaching";
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    TreatMissingData["IGNORE"] = "ignore";
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric.
 */
class Alarm extends alarm_base_1.AlarmBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo).
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation.
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *    you have both a "small margin/long period" alarm as well as a
     *    "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: mathExprHasSubmetrics(expr) ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
//# sourceMappingURL=data:application/json;base64,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