"""
scaffoldgraphnew.scripts.generate
"""

import datetime
import time

from loguru import logger

from scaffoldgraphnew import ScaffoldNetwork, ScaffoldTree, HierS
from scaffoldgraphnew.prioritization import ScaffoldRuleSet
from scaffoldgraphnew.io import tsv

from .misc import file_format

start_message = """
Running ScaffoldGraph ({command}) Generation with options:
    Input file:     {input}
    Output file:    {output}
    Maximum rings:  {max_r}
    Flatten isotopes: {isotope}
    Keep largest Fragment: {fragment}
    Discharge & Deradicalize: {discharge}
"""

stop_message = """
ScaffoldGraph Generation Complete:
    Molecules written: {molecules}
    Scaffolds written: {scaffolds}
    Molecules filtered: {filtered}
    Linear molecules: {linear}
    Time elapsed: {time}
    
Output saved @ {output}
"""


def _get_graph_cls(name):
    """Get scaffoldgraphnew class from name string."""
    if name == 'network':
        return ScaffoldNetwork
    elif name == 'tree':
        return ScaffoldTree
    elif name == 'hiers':
        return HierS
    else:
        msg = f'scaffold graph type: {name} not known'
        raise ValueError(msg)


def _maybe_ruleset(args):
    """Return a ScaffoldRuleset if specified in CLI arguments."""
    ruleset = None
    if 'ruleset' in args and args.ruleset is not None:
        filename = args.ruleset
        ruleset = ScaffoldRuleSet.from_rule_file(filename)
    return ruleset


def generate_cli(args):
    """Run scaffoldgraphnew generation for CLI utility."""
    graph_cls = _get_graph_cls(args.command)
    graph_name = graph_cls.__name__
    ruleset = _maybe_ruleset(args)

    if not args.silent:
        print(
            start_message.format(
                command=graph_name,
                input=args.input,
                output=args.output,
                max_r=args.max_rings,
                isotope=args.flatten_isotopes,
                fragment=args.keep_largest_fragment,
                discharge=args.discharge_and_deradicalize,
            )
        )

    logger.info(f'Generating {graph_name} Graph...')
    fmt, zipped = file_format(args.input)
    start = time.time()

    if fmt == 'SDF':
        sg = graph_cls.from_sdf(
            args.input,
            ring_cutoff=args.max_rings,
            progress=args.silent is False,
            zipped=zipped,
            flatten_isotopes=args.flatten_isotopes,
            keep_largest_fragment=args.keep_largest_fragment,
            discharge_and_deradicalize=args.discharge_and_deradicalize,
            prioritization_rules=ruleset,
        )
    elif fmt == 'SMI':
        sg = graph_cls.from_smiles_file(
            args.input,
            ring_cutoff=args.max_rings,
            progress=args.silent is False,
            flatten_isotopes=args.flatten_isotopes,
            keep_largest_fragment=args.keep_largest_fragment,
            discharge_and_deradicalize=args.discharge_and_deradicalize,
            prioritization_rules=ruleset,
        )
    else:
        raise ValueError('input file format is not currently supported')

    tsv.write_tsv(sg, args.output, write_ids=False)
    logger.info(f'{graph_name} Graph Generation Complete...')
    elapsed = datetime.timedelta(seconds=round(time.time() - start))
    filtered = sg.graph['num_filtered']
    linear = sg.graph['num_linear']

    if not args.silent:
        print(
            stop_message.format(
                molecules=sg.num_molecule_nodes,
                scaffolds=sg.num_scaffold_nodes,
                filtered=filtered,
                linear=linear,
                time=elapsed,
                output=args.output
            )
        )
