"""
scaffoldgraphnew.analysis.representation

Module contains general functions for scaffold analysis
"""


def get_virtual_scaffolds(scaffoldgraph, data=False, default=None):
    """Get 'virtual' scaffolds within a scaffold graph.

    Virtual scaffolds represent scaffolds that are not directly obtained from
    any molecule of the collection, but generated by the pruning process.
    Virtual scaffolds may provide promising starting points for the synthesis
    or acquisition of compounds complementing the current collection.

    Parameters
    ----------
    scaffoldgraph : ScaffoldGraph
        A ScaffoldGraph object to query
    data : str, bool, optional
        The scaffold node attribute returned in 2-tuple (n, ddict[data]).
        If True, return entire node attribute dict as (n, ddict).
        If False, return just the nodes n. The default is False.
    default : value, bool, optional
        Value used for nodes that don't have the requested attribute.
        Only relevant if data is not True or False.

    Returns
    -------
    list
        A list of scaffold node keys corresponding to virtual scaffolds.

    """
    virtual = []
    for scaffold, d in scaffoldgraph.get_scaffold_nodes(True):
        mol_count = 0
        for succ in scaffoldgraph.successors(scaffold):
            if scaffoldgraph.nodes[succ].get('type') == 'molecule':
                mol_count += 1
        if mol_count == 0:
            if data is False:
                virtual.append(scaffold)
            elif data is True:
                virtual.append((scaffold, d))
            else:
                virtual.append((scaffold, d.get(data, default)))
    return virtual


def get_singleton_scaffolds(scaffoldgraph, data=False, default=None):
    """Get singleton scaffolds within a scaffold graph.

    Singleton scaffolds represent scaffolds that are direct members of only
    one compound in the current collection.

    Parameters
    ----------
    scaffoldgraph : ScaffoldGraph
        A ScaffoldGraph object to query
    data : str, bool, optional
        The scaffold node attribute returned in 2-tuple (n, ddict[data]).
        If True, return entire node attribute dict as (n, ddict).
        If False, return just the nodes n. The default is False.
    default : value, bool, optional
        Value used for nodes that don't have the requested attribute.
        Only relevant if data is not True or False.

    Returns
    -------
    list
        A list of scaffold node keys corresponding to virtual scaffolds.

    """
    singletons = []
    for scaffold, d in scaffoldgraph.get_scaffold_nodes(True):
        mol_count = 0
        for succ in scaffoldgraph.successors(scaffold):
            if scaffoldgraph.nodes[succ].get('type') == 'molecule':
                mol_count += 1
        if mol_count == 1:
            if data is False:
                singletons.append(scaffold)
            elif data is True:
                singletons.append((scaffold, d))
            else:
                singletons.append((scaffold, d.get(data, default)))
    return singletons
