#!/usr/bin/env python3
from __future__ import annotations

import argparse
import sys

import sqlite3
import pandas


def parse_arguments() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Creates a SQLite3 database from a csv-like file"
    )
    requiredNamed = parser.add_argument_group("Required named arguments")

    requiredNamed.add_argument(
        "-t",
        "--table_name",
        help="Name of the table to be created This has to be unique for this dictionairy.",
        required=True,
    )
    requiredNamed.add_argument(
        "-f",
        "--dictionairy_file",
        help='csv-like file file location to read from. Lemmas have to be in a column named "Lemma"',
        required=True,
    )

    parser.add_argument(
        "-b",
        "--database_file",
        nargs="?",
        default="databases/main.db",
        help="File used for the SQLite Database.",
    )
    parser.add_argument(
        "-s",
        "--seperator",
        nargs="?",
        default=",",
        help='Seperator used in the database file. Tabs would be "\\t", commas ",", for example',
    )
    parser.add_argument(
        "-e",
        "--if_exists",
        choices=["fail", "replace"],
        nargs="?",
        default="fail",
        help='What to do if a table with the specified name already exists in the database. Either "fail" or "replace"',
    )
    return parser.parse_args()


def read_file(file: str, *, sep: str = " ") -> pandas.DataFrame:
    """Read file into Pandas dataframe with correct constraints.
    Trowing an understadable error when user puts in bad data"""
    try:
        return pandas.read_csv(
            file, sep=sep, index_col="Lemma", dtype=str, engine="python"
        )
    except ValueError as error:
        print(
            """Unable to read file (Make the lemmas in the specified file are in a column named \"Lemma\")

Error text generated by Python: \n""",
            error,
        )
        sys.exit()


def main():
    args: argparse.Namespace = parse_arguments()
    df: pandas.DataFrame = read_file(args.dictionairy_file, sep=args.seperator)

    try:
        database: sqlite3.Connection = sqlite3.connect(args.database_file)
        try:
            df.to_sql(args.table_name, database, if_exists=args.if_exists)
        except ValueError as error:
            print("Cannot write to SQL Database\n", error)

    except sqlite3.Error as error:
        print("SQL Error occured - ", error)

    finally:
        if database:
            database.close()


if __name__ == "__main__":
    main()
