"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log.
 *
 * @stability stable
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    FlowLogTrafficType["ALL"] = "ALL";
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs.
 *
 * @stability stable
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for.
 *
 * @stability stable
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
_a = JSII_RTTI_SYMBOL_1;
FlowLogResourceType[_a] = { fqn: "@aws-cdk/aws-ec2.FlowLogResourceType", version: "1.121.0" };
/**
 * The destination type for the flow log.
 *
 * @stability stable
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination.
     *
     * @stability stable
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination.
     *
     * @stability stable
     */
    static toS3(bucket, keyPrefix) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
            keyPrefix,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
_b = JSII_RTTI_SYMBOL_1;
FlowLogDestination[_b] = { fqn: "@aws-cdk/aws-ec2.FlowLogDestination", version: "1.121.0" };
/**
 *
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
            keyPrefix: this.props.keyPrefix,
        };
    }
}
/**
 *
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 *
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 *
 * @stability stable
 * @resource AWS::EC2::FlowLog
 */
class FlowLog extends FlowLogBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        this.keyPrefix = destinationConfig.keyPrefix;
        let logDestination = undefined;
        if (this.bucket) {
            logDestination = this.keyPrefix ? this.bucket.arnForObjects(this.keyPrefix) : this.bucket.bucketArn;
        }
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id.
     *
     * @stability stable
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
_c = JSII_RTTI_SYMBOL_1;
FlowLog[_c] = { fqn: "@aws-cdk/aws-ec2.FlowLog", version: "1.121.0" };
//# sourceMappingURL=data:application/json;base64,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