# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['secimport']

package_data = \
{'': ['*'],
 'secimport': ['profiles/*',
               'templates/*',
               'templates/actions/*',
               'templates/filters/*',
               'templates/headers/*',
               'templates/probes/*']}

install_requires = \
['PyYAML>=6.0,<7.0']

setup_kwargs = {
    'name': 'secimport',
    'version': '0.4.2',
    'description': 'A sandbox/supervisor for python modules.',
    'long_description': '# secimport\n\n<p align="center">\n <a href="https://github.com/avilum/secimport"><img src="https://user-images.githubusercontent.com/19243302/177835749-6aec7200-718e-431a-9ab5-c83c6f68565e.png" alt="secimport"></a>\n</p>\n<p align="center">\nSecure import for python modules using dtrace under the hood.<br>\n<a href="https://infosecwriteups.com/sandboxing-python-modules-in-your-code-1e590d71fc26?source=friends_link&sk=5e9a2fa4d4921af0ec94f175f7ee49f9">Medium Article</a>\n</p>\n\n`secimport` can be used to:\n- Confine/Restrict specific python modules inside your production environment.\n  - Open Source, 3rd party from unstrusted sources.\n  - Audit the flow of your python application at user-space/os/kernel level.\n- Run an entire python application under unified configuration\n  - Like `seccomp` but not limited to Linux kernels. Cross platform.\n\n\n# Quick Start\n`secimport` can be used out of the box in the following ways:\n1. Inside your code using `module = secimport.secure_import(\'module_name\', ...)`.\n    - Replacing the regular `import` statement with `secure_import`\n    - Only modules that were imported with `secure_import` will be traced.\n2. As a sandbox, by specifying the modules and their policies.\n    - Use this repository to:\n        - Generate a YAML policy from your code\n        - Compile that YAML to dscript.\n    - Use `dtrace` command to run your main python application, with your tailor-made sandbox.\n        - No need for `secure_import`, you can keep using regular `import`s\n\nFor the full list of examples, see <a href="docs/EXAMPLES.md">EXAMPLES.md</a>.\n\n# Pickle Example\n### How pickle can be exploited in your 3rd party packages:\n```python\n>>> import pickle\n>>> class Demo:\n...     def __reduce__(self):\n...         return (eval, ("__import__(\'os\').system(\'echo Exploited!\')",))\n... \n>>> pickle.dumps(Demo())\nb"\\x80\\x04\\x95F\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x8c\\x08builtins\\x94\\x8c\\x04eval\\x94\\x93\\x94\\x8c*__import__(\'os\').system(\'echo Exploited!\')\\x94\\x85\\x94R\\x94."\n>>> pickle.loads(b"\\x80\\x04\\x95F\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x8c\\x08builtins\\x94\\x8c\\x04eval\\x94\\x93\\x94\\x8c*__import__(\'os\').system(\'echo Exploited!\')\\x94\\x85\\x94R\\x94.")\nExploited!\n0\n```\nWith `secimport`, you can control such action to do whatever you want:\n```python\nIn [1]: import secimport\nIn [2]: pickle = secimport.secure_import("pickle")\nIn [3]: pickle.loads(b"\\x80\\x04\\x95F\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x8c\\x08builtins\\x94\\x8c\\x04eval\\x94\\x93\\x94\\x8c*__import__(\'os\').system(\'echo Exploited!\')\\x94\\x85\\x94R\\x94.")\n\n[1]    28027 killed     ipython\n```\nA log file is automatically created, containing everything you need to know:\n```\n$ less /tmp/.secimport/sandbox_pickle.log\n\n  @posix_spawn from /Users/avilumelsky/Downloads/Python-3.10.0/Lib/threading.py\n    DETECTED SHELL:\n        depth=8\n        sandboxed_depth=0\n        sandboxed_module=/Users/avilumelsky/Downloads/Python-3.10.0/Lib/pickle.py  \n\n    TERMINATING SHELL:\n        libsystem_kernel.dylib`__posix_spawn+0xa\n        ...\n                libsystem_kernel.dylib`__posix_spawn+0xa\n                libsystem_c.dylib`system+0x18b\n                python.exe`os_system+0xb3\n    KILLED\n:\n```\n\n## YAML Template Example\nFor a full tutorial, see <a href="docs/YAML_PROFILES.md">YAML Profiles Usage</a>\n```shell\n# An example yaml template for a sandbox.\n\nmodules:\n  requests:\n    destructive: true\n    syscall_allowlist:\n      - write\n      - ioctl\n      ...\n      - stat64\n  fastapi:\n    destructive: true\n    syscall_allowlist:\n      - bind\n      - fchmod\n      ...\n      - stat64\n  uvicorn:\n    destructive: true\n    syscall_allowlist:\n      - getpeername\n      - getpgrp\n      ...\n      - stat64\n\n```\n\n## Python Processing Example\n```python\nPython 3.10.0 (default, May  2 2022, 21:43:20) [Clang 13.0.0 (clang-1300.0.27.3)] on darwin\nType "help", "copyright", "credits" or "license" for more information.\n\n# Let\'s import subprocess module, limiting it\'s syscall access.\n>>> import secimport\n>>> subprocess = secimport.secure_import("subprocess")\n\n# Let\'s import os \n>>> import os\n>>> os.system("ps")\n  PID TTY           TIME CMD\n 2022 ttys000    0:00.61 /bin/zsh -l\n50092 ttys001    0:04.66 /bin/zsh -l\n75860 ttys001    0:00.13 python\n0\n# It worked as expected, returning exit code 0.\n\n\n# Now, let\'s try to invoke the same logic using a different module, "subprocess", that was imported using secure_import:\n>>> subprocess.check_call(\'ps\')\n[1]    75860 killed     python\n\n# Damn! That\'s cool.\n```\n\n- The dtrace profile for the module is saved under:\n  -  `/tmp/.secimport/sandbox_subprocess.d`:\n- The log file for this module is under\n  -  `/tmp/.secimport/sandbox_subprocess.log`:\n        ```shell\n        ...\n\n        (OPENING SHELL using posix_spawn): (pid 75860) (thread 344676) (user 501) (python module: <stdin>) (probe mod=, name=entry, prov=syscall func=posix_spawn) /bin/sh \n            #posix_spawn,\n\n        (TOUCHING FILESYSTEM): write(140339021606912) from thread 344676\n                    libsystem_kernel.dylib`__fork+0xb\n                    _posixsubprocess.cpython-310-darwin.so`do_fork_exec+0x29\n                    _posixsubprocess.cpython-310-darwin.so`subprocess_fork_exec+0x71f\n                    python.exe`cfunction_call+0x86\n        killing...\n        killed.\n        ```\n\n## Shell Blocking Example\n```python\n# example.py - Executes code upon import;\nimport os;\n\nos.system(\'Hello World!\');\n```\n```python\n# production.py - Your production code\nfrom secimport import secure_import \n\nexample = secure_import(\'example\', allow_shells=False)\n```\nLet\'s run the  and see what happens:\n```\n(root) sh-3.2#  export PYTHONPATH=$(pwd)/src:$(pwd)/examples:$(pwd):$PYTHONPATH\n(root) sh-3.2#  python examples/production.py \nSuccessfully compiled dtrace profile:  /tmp/.secimport/sandbox_example.d\nKilled: 9\n```\n- We imported `example` with limited capabilities.\n- If a syscall like `spawn/exec/fork/forkexec` will be executed\n  - The process will be `kill`ed with `-9` signal.\n\n## Network Blocking Example\n```\n>>> import requests\n>>> requests.get(\'https://google.com\')\n<Response [200]>\n  \n\n>>> from secimport import secure_import\n>>> requests = secure_import(\'requests\', allow_networking=False)\n\n# The next call should kill the process,\n# because we disallowed networking for the requests module.\n>>> requests.get(\'https://google.com\')\n[1]    86664 killed\n```\n### Requirements\nThe only requirement is a Python interpreter that was built with --with-dtrace.\n  - See <a href="docs/INSTALL.md">INSTALL.md</a> for a detailed setup from scratch.\n- pip\n  - `python3 -m pip install secimport`\n- Poetry\n  - `python3 -m pip install poetry && python3 -m poetry build`\n<br>\n\n### Tests\n`python -m pytest`\n\n\n### Log4Shell as an example\nNot related for python, but for the sake of explanation (Equivilant Demo soon).\n- <a href="https://cve.mitre.org/cgi-bin/cvename.cgi?name=cve-2021-44228">Log4Shell - CVE-2021-44228</a>\n  - Let\'s say we want to block `log4j` from doing crazy things.\n  - In the following import we deny `log4j` from opening an LDAP connection / shell:\n    - `log4j = secure_import(\'log4j\', allow_shells=False, allow_networking=False)`\n  - This would disable `log4j` from opening sockets and execute commands, IN THE KERNEL.\n  - You can choose any policy you like for any module.\n<br><br>\n\n\n# Useful References\n- <a href="docs/EXAMPLES.md">Examples</a>\n- <a href="docs/TRACING_PROCESSES.md">Tracing Guides</a>\n- <a href="docs/FAQ.md">F.A.Q</a>\n- <a href="docs/INSTALL.md">Installation</a>\n- <a href="docs/MAC_OS_USERS.md">Mac OS Users</a> - Disabling SIP for dtrace\n- https://www.brendangregg.com/DTrace/DTrace-cheatsheet.pdf\n<br><br>\n\n## TODO:\n- ✔️ Allow/Block list configuration\n- ✔️ Create a .yaml configuration per module in the code\n  - ✔️ Use secimport to compile that yml\n  - ✔️ Create a single dcript policy\n  - ✔️ Run an application with that policy using dtrace, without using `secure_import`\n- Node support (dtrace hooks)\n- Go support (dtrace hooks)\n- Use current_module_str together with thread ID\n',
    'author': 'Avi Lumelsky',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/avilum/secimport',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
