# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['handpick']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'handpick',
    'version': '0.13.1',
    'description': 'Traverse nested data structures.',
    'long_description': '|ukraine|\n\n|version| |tests| |coverage| |pyversions| |pre-commit| |black| |bandit|\n\n==========\n Handpick\n==========\n\nHandpick is a tool to work with nested data structures.\n\n\nInstallation\n============\n\n.. code::\n\n    pip install handpick\n\n\nQuick introduction\n==================\n\n\nThe ``pick`` function\n---------------------\n\nThe `pick`_ generator function performs the recursive traversal\nof a nested data structure and picks all objects that meet certain\ncriteria provided in the form of a predicate function.\nPicked objects are retrieved lazily by an iterator.\n\n\nSimple predicate functions\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThe predicate function is passed to ``pick`` as the ``predicate``\nargument. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    def is_non_empty_string(obj):\n        return isinstance(obj, str) and obj\n\n    data = [[1, ""], [-2, ["foo", 3.0]], -4, "bar"]\n\n.. code::\n\n    >>> for s in pick(data, predicate=is_non_empty_string):\n    ...     print(s)\n    ...\n    foo\n    bar\n\n\nHandling dictionary keys\n~~~~~~~~~~~~~~~~~~~~~~~~\n\nWhen inspecting dictionaries or other mappings, you can configure\nwhether or not ``pick`` will inspect dictionary keys using the\n``dict_keys`` keyword argument. Default is False, which means only\ndictionary values are inspected. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = {"foo": {"name": "foo"}, "bar": {"name": "bar"}}\n\n.. code::\n\n    >>> for s in pick(data, predicate=lambda obj: "a" in obj):\n    ...     print(s)\n    ...\n    bar\n    >>> for s in pick(data, predicate=lambda obj: "a" in obj, dict_keys=True):\n    ...     print(s)\n    ...\n    name\n    bar\n    name\n    bar\n\n\nPredicates\n----------\n\n\nThe ``Predicate`` decorator\n~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThe `Predicate`_ decorator wraps a function in an object that can be\ncombined with other predicates using the operators ``&`` (and) and\n``|`` (or), as well as negated using the operator ``~`` (not).\n\n\nCombining predicates\n~~~~~~~~~~~~~~~~~~~~\n\nFor example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def is_integer(obj):\n        return isinstance(obj, int)\n\n    @Predicate\n    def is_even(number):\n        return number % 2 == 0\n\n    data = [[4, [5.0, 1], 3.0], [[15, []], {17: [7, [8], 0]}]]\n\n    # compound predicate\n    odd_int = is_integer & ~is_even\n\n.. code::\n\n    >>> for n in pick(data, predicate=odd_int):\n    ...     print(n)\n    ...\n    1\n    15\n    7\n\n\nCombining predicates with functions\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nIn addition, the ``&`` and ``|`` operations are supported between\npredicates and regular undecorated functions. For example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def is_list(obj):\n        return isinstance(obj, list)\n\n    data = [("1", [2]), {("x",): [(3, [4]), "5"]}, ["x", ["6"]], {7: ("x",)}]\n\n    # compound predicate\n    short_list = (lambda obj: len(obj) < 2) & is_list\n\n.. code::\n\n    >>> for l in pick(data, predicate=short_list):\n    ...     print(l)\n    ...\n    [2]\n    [4]\n    [\'6\']\n\n\nSuppressing errors\n~~~~~~~~~~~~~~~~~~\n\nThe important thing to note is that when the predicate\'s underlying\nfunction raises an exception, the exception is suppressed and the predicate\nreturns False. In other words, it is assumed that the object in question does\nnot meet the picking criteria. For example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def above_zero(number):\n        return number > 0\n\n.. code::\n\n    >>> above_zero(1)\n    True\n    >>> above_zero("a")\n    False\n    >>> for n in pick([[1, "Py", -2], [None, 3.0]], predicate=above_zero):\n    ...     print(n)\n    ...\n    1\n    3.0\n\nIn the example above, several lists and strings were internally compared to ``0``\nbut no ``TypeError`` propagated up to the code that called ``above_zero``.\n\n\nPredicate factories\n~~~~~~~~~~~~~~~~~~~\n\nThe `is_type`_ function can be used to create\npredicates based on an object\'s type. For example:\n\n.. code-block:: python\n\n    from handpick import pick, is_type\n\n    data = [[1.0, [2, True]], [False, [3]], ["4"]]\n\n    strictly_int = is_type(int) & ~is_type(bool)\n\n.. code::\n\n    >>> for n in pick(data, predicate=strictly_int):\n    ...     print(n)\n    ...\n    2\n    3\n\n\nBuilt-in predicates\n~~~~~~~~~~~~~~~~~~~\n\nHandpick provides some predefined predicates to be used in common\nscenarios. For example:\n\n.. code-block:: python\n\n    from handpick import pick, NUM_STR\n\n    data = {"id": "01353", "price": 15.42, "quantity": 68, "year": "2011"}\n\n    # pick strings that can be cast to numbers\n    numeric_strings = pick(data, predicate=NUM_STR)\n\n.. code::\n\n    >>> for s in numeric_strings:\n    ...     print(s)\n    ...\n    01353\n    2011\n\n\nUseful functions\n----------------\n\n\nThe ``values_for_key`` function\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nWhen inspecting data structures that contain dictionaries or other\nmappings, you can use `values_for_key`_ to retrieve values associated with\na specific key, regardless of the nested depth in which these values\nare stored. Values are retrieved lazily by an iterator. For example:\n\n.. code-block:: python\n\n    from handpick import values_for_key\n\n    data = {\n        "node_id": 4,\n        "child_nodes": [\n            {\n                "node_id": 8,\n                "child_nodes": [\n                    {\n                        "node_id": 16,\n                    },\n                ],\n            },\n            {\n                "id": 9,\n            },\n        ],\n    }\n\n.. code::\n\n    >>> for i in values_for_key(data, key="node_id"):\n    ...     print(i)\n    ...\n    4\n    8\n    16\n\nMultiple keys may be specified at a time. For example:\n\n.. code::\n\n    >>> for i in values_for_key(data, key=["node_id", "id"]):\n    ...     print(i)\n    ...\n    4\n    8\n    16\n    9\n\n\nThe ``max_depth`` function\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThis function returns the maximum nested depth of a data structure. For\nexample:\n\n.. code::\n\n    >>> from handpick import max_depth\n    >>> max_depth([0, [1, [2]]])\n    2\n    >>> max_depth({0: {1: {2: {3: {4: 4}}}}})\n    4\n\n**Note:** Just like non-empty collections, empty collections constitute\nanother level of nested depth. For example:\n\n.. code::\n\n    >>> max_depth([0, [1, []]])\n    2\n\n\nRecipes\n=======\n\n\nFlattening nested data\n----------------------\n\nUse the `pick`_ function, omitting the ``predicate`` argument and passing\n``collections=False``. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = [[], [0], [[[], 1], [2, [3, [4]], []], [5]]]\n\n.. code::\n\n    >>> list(pick(data, collections=False))\n    [0, 1, 2, 3, 4, 5]\n\n\nAPI reference\n=============\n\npick\n----\n\n*handpick.pick(data, predicate=None, *, collections=True, dict_keys=False, strings=False, bytes_like=False)*\n\nPick objects from ``data`` based on ``predicate``.\n\nTraverse ``data`` recursively and yield all objects for which\n``predicate(obj)`` is True or truthy. ``data`` should be an iterable\ncollection.\n\n``predicate`` must be callable, must take one argument, and should\nreturn a Boolean value. If ``predicate`` is omitted or None, all objects\nare picked.\n\nBy default, collections of other objects are yielded just like any\nother objects. To exclude collections, pass ``collections=False``.\n\nWhen traversing a mapping, only its values are inspected by default.\nTo inspect both keys and values of mappings, pass ``dict_keys=True``.\n\nBy default, strings are not treated as collections of other objects\nand therefore not iterated by the recursive algorithm. This can be\nchanged by passing ``strings=True``. Empty strings and strings of\nlength 1 are never iterated.\n\nBy default, bytes-like sequences (bytes and bytearrays) are not\ntreated as collections of other objects and therefore not iterated\nby the recursive algorithm. This can be changed by passing\n``bytes_like=True``.\n\nPredicate\n---------\n\n*@handpick.Predicate(func=None, *, suppressed_errors=(TypeError, ValueError, LookupError, AttributeError))*\n\nDecorator wrapping a function in a predicate object.\n\nThe decorated function can be combined with other predicates using\nthe operators ``&`` (and) and ``|`` (or), as well as negated using the\noperator ``~`` (not).\n\n``suppressed_errors`` can be used to customize which exception classes\nwill be suppressed by the predicate.\n\nPredicate objects are intended to be used as the ``predicate``\nargument to the ``pick`` function.\n\nis_type\n-------\n\n*handpick.is_type(type_or_types)*\n\nPredicate factory. Return a predicate that returns True if\nobject is an instance of specified type(s).\n\n``type_or_types`` must be a type or tuple of types.\n\nno_error\n--------\n\n*handpick.no_error(func)*\n\nPredicate factory. Return a predicate that returns True if ``func``\ncan be applied on object without an exception being raised,\nFalse otherwise.\n\nINT_STR\n-------\n\n*handpick.INT_STR*\n\nPredicate that returns True for strings that can be converted\nto int.\n\nFLOAT_STR\n---------\n\n*handpick.FLOAT_STR*\n\nPredicate that returns True for strings that can be converted\nto float.\n\nNUM_STR\n-------\n\n*handpick.NUM_STR*\n\nPredicate that returns True for strings that can be converted\nto a number (i.e. an int, float or complex).\n\nvalues_for_key\n--------------\n\n*handpick.values_for_key(data, key)*\n\nPick values associated with a specific key.\n\nTraverse ``data`` recursively and yield a sequence of dictionary\nvalues that are mapped to ``key``. ``key`` may be a list of multiple\nkeys.\n\nmax_depth\n---------\n\n*handpick.max_depth(data)*\n\nReturn maximum nested depth of ``data``.\n\n``data`` should be an iterable collection. Depth is counted from zero,\ni.e. the direct elements of ``data`` are in depth 0.\n\n\n.. |version| image:: https://img.shields.io/pypi/v/handpick\n    :target: https://pypi.org/project/handpick\n.. |ukraine| image:: https://raw.githubusercontent.com/vshymanskyy/StandWithUkraine/main/badges/StandWithUkraine.svg\n    :target: https://stand-with-ukraine.pp.ua\n.. |tests| image:: https://github.com/mportesdev/handpick/actions/workflows/tests.yml/badge.svg\n    :target: https://github.com/mportesdev/handpick/actions\n.. |coverage| image:: https://img.shields.io/codecov/c/gh/mportesdev/handpick\n    :target: https://codecov.io/gh/mportesdev/handpick\n.. |pyversions| image:: https://img.shields.io/pypi/pyversions/handpick\n    :target: https://pypi.org/project/handpick\n.. |pre-commit| image:: https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit\n    :target: https://github.com/pre-commit/pre-commit\n.. |black| image:: https://img.shields.io/badge/code%20style-black-000000.svg\n   :target: https://github.com/psf/black\n.. |bandit| image:: https://img.shields.io/badge/security-bandit-yellow.svg\n    :target: https://github.com/PyCQA/bandit\n',
    'author': 'Michal Porteš',
    'author_email': 'michalportes1@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/mportesdev/handpick',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
