import os
from subprocess import Popen, PIPE, STDOUT
import logging

import mlflow
import mlflow.version
from mlflow.utils.file_utils import TempDir, _copy_project
from mlflow.utils.logging_utils import eprint

_logger = logging.getLogger(__name__)

DISABLE_ENV_CREATION = "MLFLOW_DISABLE_ENV_CREATION"

_DOCKERFILE_TEMPLATE = """
# Build an image that can serve mlflow models.
FROM ubuntu:16.04

RUN apt-get -y update && apt-get install -y --no-install-recommends \
         wget \
         curl \
         nginx \
         ca-certificates \
         bzip2 \
         build-essential \
         cmake \
         openjdk-8-jdk \
         git-core \
         maven \
         tree \
         python3-dev \
         python3-setuptools \
         python3-pip \
    && rm -rf /var/lib/apt/lists/*

# Download and setup miniconda
RUN ln -s /usr/bin/pip3 /usr/bin/pip
RUN ln -s /usr/bin/python3 /usr/bin/python
RUN curl https://repo.continuum.io/miniconda/Miniconda3-latest-Linux-x86_64.sh >> miniconda.sh
RUN bash ./miniconda.sh -b -p /miniconda; rm ./miniconda.sh;
ENV PATH="/miniconda/bin:$PATH"
ENV JAVA_HOME=/usr/lib/jvm/java-8-openjdk-amd64
ENV GUNICORN_CMD_ARGS="--timeout 60 -k gevent"
# Set up the program in the image
WORKDIR /opt/mlflow

{install_mlflow}

{custom_setup_steps}
{entrypoint}
"""


def _get_mlflow_install_step(dockerfile_context_dir, mlflow_home):
    """
    Get docker build commands for installing MLflow given a Docker context dir and optional source
    directory
    """
    if mlflow_home:
        mlflow_dir = _copy_project(
            src_path=mlflow_home, dst_path=dockerfile_context_dir
        )
        return (
            "COPY {mlflow_dir} /opt/mlflow\n"
            "RUN pip3 install /opt/mlflow\n"
            "RUN cd /opt/mlflow/mlflow/java/scoring && "
            "mvn --batch-mode package -DskipTests && "
            "mkdir -p /opt/java/jars && "
            "mv /opt/mlflow/mlflow/java/scoring/target/"
            "mlflow-scoring-*-with-dependencies.jar /opt/java/jars\n"
        ).format(mlflow_dir=mlflow_dir)
    else:
        return (
            "RUN pip3 install mlflow-sagemaker=={version}\n"
            "RUN mvn "
            " --batch-mode dependency:copy"
            " -Dartifact=org.mlflow:mlflow-scoring:{upstream_version}:pom"
            " -DoutputDirectory=/opt/java\n"
            "RUN mvn "
            " --batch-mode dependency:copy"
            " -Dartifact=org.mlflow:mlflow-scoring:{upstream_version}:jar"
            " -DoutputDirectory=/opt/java/jars\n"
            "RUN cp /opt/java/mlflow-scoring-{upstream_version}.pom /opt/java/pom.xml\n"
            "RUN cd /opt/java && mvn "
            "--batch-mode dependency:copy-dependencies -DoutputDirectory=/opt/java/jars\n"
        ).format(version=mlflow.version.VERSION, upstream_version='1.5.0')


def _build_image(
    image_name, entrypoint, mlflow_home=None, custom_setup_steps_hook=None, no_cache=False
):
    """
    Build an MLflow Docker image that can be used to serve a
    The image is built locally and it requires Docker to run.

    :param image_name: Docker image name.
    :param entry_point: String containing ENTRYPOINT directive for docker image
    :param mlflow_home: (Optional) Path to a local copy of the MLflow GitHub repository.
                        If specified, the image will install MLflow from this directory.
                        If None, it will install MLflow from pip.
    :param custom_setup_steps_hook: (Optional) Single-argument function that takes the string path
           of a dockerfile context directory and returns a string containing Dockerfile commands to
           run during the image build step.
   :param no_cache: (Optional) Remove cache before building the image.
    """
    mlflow_home = os.path.abspath(mlflow_home) if mlflow_home else None
    with TempDir() as tmp:
        cwd = tmp.path()
        install_mlflow = _get_mlflow_install_step(cwd, mlflow_home)
        custom_setup_steps = (
            custom_setup_steps_hook(cwd) if custom_setup_steps_hook else ""
        )
        with open(os.path.join(cwd, "Dockerfile"), "w") as f:
            f.write(
                _DOCKERFILE_TEMPLATE.format(
                    install_mlflow=install_mlflow,
                    custom_setup_steps=custom_setup_steps,
                    entrypoint=entrypoint,
                )
            )
        _logger.info("Building docker image with name %s", image_name)
        os.system("find {cwd}/".format(cwd=cwd))

        if no_cache:
            cmd = ["docker", "build", "--no-cache", "-t", image_name, "-f", "Dockerfile", "."]
        else:
            cmd = ["docker", "build", "-t", image_name, "-f", "Dockerfile", "."]
        proc = Popen(
            cmd,
            cwd=cwd,
            stdout=PIPE,
            stderr=STDOUT,
            universal_newlines=True,
        )
        for x in iter(proc.stdout.readline, ""):
            eprint(x, end="")
