import abc
import glob
import gzip
import multiprocessing
import os
import queue
import re
import threading
import uuid

from functools import cached_property
from collections import UserDict

from searchkit.log import log

RESULTS_QUEUES = {}
RESULTS_QUEUE_TIMEOUT = 60


class FileSearchException(Exception):
    def __init__(self, msg):
        self.msg = msg


class SearchDefBase(object):

    def __init__(self, constraints=None):
        self.id
        self.constraints = {c.id: c for c in constraints or {}}

    @cached_property
    def id(self):
        """
        A unique identifier for this search definition.
        """
        return str(uuid.uuid4())


class SearchDef(SearchDefBase):

    def __init__(self, pattern, tag=None, hint=None,
                 store_result_contents=True, **kwargs):
        """
        Add a search definition.

        @param pattern: regex pattern or list of patterns to search for
        @param tag: optional user-friendly identifier for this search term
        @param hint: pre-search term to speed things up
        @param store_result_contents: by default the content of a search result
                                      is saved but if it is not needed this
                                      can be set to False. This effectively
                                      makes the result True/False.
        """
        if type(pattern) != list:
            self.patterns = [re.compile(pattern)]
        else:
            self.patterns = []
            for _pattern in pattern:
                self.patterns.append(re.compile(_pattern))

        self.store_result_contents = store_result_contents
        self.tag = tag
        if hint:
            self.hint = re.compile(hint)
        else:
            self.hint = None

        # do this last
        super().__init__(**kwargs)

    def run(self, line):
        """Execute search patterns against line and return first match."""
        if self.hint:
            ret = self.hint.search(line)
            if not ret:
                return None

        ret = None
        for pattern in self.patterns:
            ret = pattern.match(line)
            if ret:
                return ret

        return ret


class SequenceSearchDef(SearchDefBase):

    def __init__(self, start, tag, end=None, body=None, **kwargs):
        """
        Define search for sequences. A sequence must match a start and end with
        optional body in between. If no end defined, the sequence ends with
        the start of the next or EOF.

        NOTE: sequences must not overlap. This is therefore not suitable for
        finding sequences generated by parallel/concurrent tasks.

        @param start: SearchDef object for matching start
        @param tag: tag used to identify this sequence definition
        @param end: optional SearchDef object for matching end
        @param body: optional SearchDef object for matching body
        """
        self.s_start = start
        self.s_end = end
        self.s_body = body
        self.tag = tag
        self._mark = None
        # Each section identified gets its own id. Since each file is processed
        # using a separate process and memory is not shared, these values must
        # be unique to avoid collisions when results are aggregated.
        self._section_id = None
        # do this last
        super().__init__(**kwargs)

    @property
    def start_tag(self):
        """Tag used to identify start of section"""
        return "{}-start".format(self.tag)

    @property
    def end_tag(self):
        """Tag used to identify end of section"""
        return "{}-end".format(self.tag)

    @property
    def body_tag(self):
        """Tag used to identify body of section"""
        return "{}-body".format(self.tag)

    @property
    def section_id(self):
        """ ID of current section. A new id should be set after each
        completed section. """
        return self._section_id

    @property
    def started(self):
        """Indicate a section sequence has been started."""
        return self._mark == 1

    def start(self):
        """Indicate that a sequence start has been detected."""
        self._section_id = str(uuid.uuid4())
        self._mark = 1

    def reset(self):
        """Used to restart a section. This is used e.g. if the start
        expression matches midway through a sequence (and before the end).
        """
        self._mark = 0

    def stop(self):
        """Indicate that a sequence is complete."""
        self._mark = 0
        self._section_id = str(uuid.uuid4())


class SearchResultPart(object):

    def __init__(self, index, value):
        self.index = index
        self.value = value


class SearchResult(object):

    def __init__(self, linenumber, source_id, result, search_term_tag=None,
                 section_id=None, sequence_obj_id=None, store_contents=True):
        """
        @param linenumber: line number that produced a match
        @param source_id: data source id
        @param result: python.re match object
        @param search_term_tag: SearchDef object tag
        @param section_id: SequenceSearchDef object section id
        @param sequence_obj_id: SequenceSearchDef object unique id
        """
        self.tag = search_term_tag
        self.source_id = source_id
        self.linenumber = linenumber
        self._parts = {}
        self.sequence_obj_id = sequence_obj_id
        self.section_id = section_id

        if not store_contents:
            log.debug("store_contents is False - skipping save")
            return

        num_groups = len(result.groups())
        # NOTE: this does not include group(0)
        if num_groups:
            # To reduce memory footprint, don't store group(0) i.e. the whole
            # line, if there are actual groups in the result.
            for i in range(1, num_groups + 1):
                self._add(i, result.group(i))
        else:
            log.debug("saving full search result which can lead to high "
                      "memory usage")
            self._add(0, result.group(0))

    def _add(self, index, value):
        self._parts[index] = SearchResultPart(index, value)

    def get(self, index):
        """Retrieve a result part by its index."""
        if index not in self._parts:
            return None

        return self._parts[index].value

    def __len__(self):
        return len(self._parts)

    def __iter__(self):
        for idx in sorted(self._parts.keys()):
            yield self._parts[idx].value

    def __repr__(self):
        r_list = ["{}={}".format(k, v.value) for k, v in self._parts.items()]
        return "ln:{} ".format(self.linenumber) + ", ".join(r_list)


class SearchResultsCollection(object):

    def __init__(self, search_catalog):
        self.search_catalog = search_catalog
        self.reset()

    def __len__(self):
        _count = 0
        for f in self.files:
            _count += len(self.find_by_path(f))

        return _count

    @property
    def files(self):
        return list(self._results.keys())

    def reset(self):
        self._iter_idx = 0
        self._results = {}

    def add(self, result):
        # resolve
        path = self.search_catalog.source_id_to_path(result.source_id)
        if path not in self._results:
            self._results[path] = [result]
        else:
            self._results[path].append(result)

    def find_by_path(self, path):
        if path not in self._results:
            return []

        return self._results[path]

    def find_by_tag(self, tag, path=None, sequence_obj_id=None):
        """Return all result tagged with tag.

        If no path is provided tagged results from all paths are returned.
        """
        if path:
            paths = [path]
        else:
            paths = list(self._results.keys())

        results = []
        for path in paths:
            for result in self._results.get(path, []):
                if sequence_obj_id is None:
                    if result.tag == tag:
                        results.append(result)
                else:
                    if (result.tag == tag and
                            result.sequence_obj_id == sequence_obj_id):
                        results.append(result)

        return results

    def find_sequence_sections(self, sequence_obj, path=None):
        """Return results of running the given sequence search.

        Returns a dictionary keyed by section id where each is a list of
        results for that section with start, body, end etc.
        """
        _results = []
        sections = {}
        _results += self.find_by_tag(tag=sequence_obj.start_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        _results += self.find_by_tag(tag=sequence_obj.body_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        _results += self.find_by_tag(tag=sequence_obj.end_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        for r in _results:
            if r.section_id in sections:
                sections[r.section_id].append(r)
            else:
                sections[r.section_id] = [r]

        return sections

    def __iter__(self):
        return iter(self._results.items())


class LogrotateLogSort(object):

    def __call__(self, fname):
        """
        Sort contents of a directory by passing the function as the key to a
        list sort. Directory is expected to contain logfiles with extensions
        used by logrotate e.g. .log, .log.1, .log.2.gz etc.
        """
        filters = [r"\S+\.log$",
                   r"\S+\.log\.(\d+)$",
                   r"\S+\.log\.(\d+)\.gz?$"]
        for filter in filters:
            ret = re.compile(filter).match(fname)
            if ret:
                break

        # files that don't follow logrotate naming format go to the end.
        if not ret:
            # put at the end
            return 100000

        if len(ret.groups()) == 0:
            return 0

        return int(ret.group(1))


class SearchCatalog(object):
    def __init__(self, max_logrotate_depth=7):
        self.max_logrotate_depth = max_logrotate_depth
        self.user_paths = {}
        self._source_ids = {}

    def register(self, search, user_path):
        if user_path in self.user_paths:
            self.user_paths[user_path].append(search)
        else:
            self.user_paths[user_path] = [search]

    def _filtered_dir(self, contents, max_logrotate_depth=7):
        """ Filter contents of a directory. Directories are ignored and if any
        files look like logrotated log files they are sorted and only
        max_logrotate_depth are kept.
        """
        logrotated = {}
        new_contents = []
        for path in contents:
            if not os.path.isfile(path):
                continue

            ret = re.compile(r"(\S+)\.log\S*").match(path)
            if not ret:
                new_contents.append(path)
                continue

            fnamepfix = ret.group(1)
            if path.endswith('.log'):
                new_contents.append(fnamepfix + '.log')
            else:
                if fnamepfix not in logrotated:
                    logrotated[fnamepfix] = [path]
                else:
                    logrotated[fnamepfix].append(path)

        limit = max_logrotate_depth
        for logrotated in logrotated.values():
            capped = sorted(logrotated,
                            key=LogrotateLogSort())[:limit]
            new_contents += capped

        return new_contents

    def _expand_path(self, path):
        if os.path.isfile(path):
            return [path]
        elif os.path.isdir(path):
            return self._filtered_dir(os.listdir(path),
                                      self.max_logrotate_depth)

        return self._filtered_dir(glob.glob(path), self.max_logrotate_depth)

    def source_id_to_path(self, s_id):
        try:
            return self._source_ids[s_id]
        except KeyError:
            log.exception("ALL PATHS:")
            log.error('\n'.join(list(self._source_ids.keys())))

    def _get_source_id(self, path):
        for id, _path in self._source_ids.items():
            if _path == path:
                return id

        s_id = str(uuid.uuid4())
        while s_id in self._source_ids:
            log.error("source id %s already exists - trying again", s_id)
            s_id = str(uuid.uuid4())

        log.debug("path=%s source_id=%s", path, s_id)
        self._source_ids[s_id] = path
        return s_id

    def __iter__(self):
        for user_path, searches in self.user_paths.items():
            for path in self._expand_path(user_path):
                yield {'user_path': user_path,
                       'path': path,
                       'source_id': self._get_source_id(path),
                       'searches': searches}


class SearchTask(object):

    def __init__(self, info, lock, constraints_manager, results_queue_id):
        self.info = info
        self.lock = lock
        self.stats = SearchTaskStats()
        self.constraints_manager = constraints_manager
        self.results_queue_id = results_queue_id

    @cached_property
    def search_defs_conditional(self):
        return [s_term for s_term in self.info['searches']
                if s_term.constraints]

    @cached_property
    def search_defs(self):
        all = {s_term: True for s_term in self.info['searches']}
        for s_term in all:
            if s_term in self.search_defs_conditional:
                all[s_term] = False

        return all

    def put_result(self, result):
        self.stats['results'] += 1
        max_tries = 10
        _queue = RESULTS_QUEUES[self.results_queue_id]
        while max_tries > 0:
            try:
                _queue.put(result, timeout=RESULTS_QUEUE_TIMEOUT)
                break
            except queue.Full:
                log.debug("search task queue for '%s' is full even after "
                          "waiting %ss - trying again", self.info['path'],
                          RESULTS_QUEUE_TIMEOUT)
                max_tries -= 1

    def _simple_search(self, s_term, line, ln):
        """
        Perform a simple search on line.
        """
        ret = s_term.run(line)
        if not ret:
            return

        self.put_result(SearchResult(
                            ln, self.info['source_id'], ret, s_term.tag,
                            store_contents=s_term.store_result_contents))

    def _sequence_search(self, s_term, line, ln, sequence_results):
        """
        Perform a sequence search on line.
        """
        # if the ending is defined and we match a start while
        # already in a section, we start again.
        if s_term.s_end:
            ret = s_term.s_start.run(line)
            if s_term.started:
                if ret:
                    # reset and start again
                    if sequence_results:
                        del sequence_results[s_term.id]

                    s_term.reset()
                else:
                    ret = s_term.s_end.run(line)
        else:
            ret = s_term.s_start.run(line)

        if ret:
            section_id = None
            sequence_obj_id = None
            tag = s_term.tag
            store_contents = True

            if not s_term.started:
                tag = s_term.start_tag
                s_term.start()
                section_id = s_term.section_id
            else:
                tag = s_term.end_tag
                section_id = s_term.section_id
                s_term.stop()
                # if no end is defined then we don't bother storing
                # the result, just complete the section and start
                # the next.
                if s_term.s_end is None:
                    tag = s_term.start_tag
                    s_term.start()
                    section_id = s_term.section_id

            sequence_obj_id = s_term.id
            r = SearchResult(ln, self.info['source_id'], ret, tag,
                             section_id=section_id,
                             sequence_obj_id=sequence_obj_id,
                             store_contents=store_contents)
            if s_term.id not in sequence_results:
                sequence_results[s_term.id] = [r]
            else:
                sequence_results[s_term.id].append(r)
        else:
            if s_term.started and s_term.s_body:
                ret = s_term.s_body.run(line)
                if ret:
                    r = SearchResult(ln, self.info['source_id'], ret,
                                     s_term.body_tag,
                                     section_id=s_term.section_id,
                                     sequence_obj_id=s_term.id)
                    sequence_results[s_term.id].append(r)

        return sequence_results

    def _process_sequence_results(self, sequence_results, current_ln):
        """
        @param sequence_results: list of sequence search results
        @param current_ln: number of the last line to be read from file
        """
        # If a sequence ending definition is provided and we reached EOF
        # while a sequence is started, complete the sequence if s_end
        # matches an empty string. If s_end is not defined we just go ahead
        # and complete the section.
        filter_section_id = {}
        for s_term in self.search_defs:
            if type(s_term) == SequenceSearchDef:
                if s_term.started:
                    if s_term.s_end is None:
                        s_term.stop()
                    else:
                        ret = s_term.s_end.run("")
                        if ret:
                            section_id = s_term.section_id
                            s_term.stop()
                            tag = s_term.end_tag
                            r = SearchResult(current_ln + 1,
                                             self.info['source_id'], ret,
                                             tag,
                                             section_id=section_id,
                                             sequence_obj_id=s_term.id)
                        else:
                            if s_term.id not in filter_section_id:
                                filter_section_id[s_term.id] = []

                            filter_section_id[s_term.id].append(
                                s_term.section_id)

        # Now add sequence results to main results list, excluding any
        # incomplete sections.
        for s_results in sequence_results.values():
            for r in s_results:
                if filter_section_id:
                    seq_id = r.sequence_obj_id
                    if (seq_id is not None and
                            seq_id in filter_section_id):
                        if r.section_id in filter_section_id[seq_id]:
                            continue

                self.put_result(r)

    def _run_search(self, fd):
        """
        @param fd: open file descriptor
        """
        self.stats.reset()
        sequence_results = {}
        offset = self.constraints_manager.apply_global(self.info['user_path'],
                                                       fd)
        log.debug("starting search of %s (offset=%s, pos=%s)", fd.name, offset,
                  fd.tell())
        runnable = {s.id: _runnable
                    for s, _runnable in self.search_defs.items()}
        # NOTE: line numbers start at 1 hence offset + 1
        for ln, line in enumerate(fd, start=offset + 1):
            self.stats['lines_searched'] += 1
            if type(line) == bytes:
                line = line.decode("utf-8")

            for s_term in self.search_defs:
                if not runnable[s_term.id]:
                    if not self.constraints_manager.apply_single(s_term, line):
                        continue

                    # enable from here on in
                    runnable[s_term.id] = True

                if type(s_term) == SequenceSearchDef:
                    sequence_results = self._sequence_search(s_term, line, ln,
                                                             sequence_results)
                else:
                    self._simple_search(s_term, line, ln)

        if sequence_results:
            self._process_sequence_results(sequence_results, ln)

        log.debug("completed search of %s lines", self.stats['lines_searched'])
        if self.search_defs_conditional:
            msg = "constraints stats {}:".format(fd.name)
            for sd in self.search_defs_conditional:
                if sd.constraints:
                    for c in sd.constraints.values():
                        msg += "\n  id={}: {}".format(c.id, c.stats())

            log.debug(msg)

        return self.stats

    def failed(self, exc):
        """ This should be called if the task failed to execute. """
        log.error("search task failed for path=%s with exception %s",
                  self.info['path'], exc)

    def execute(self):
        path = self.info['path']
        if os.path.getsize(path) == 0:
            log.debug("filesearcher: zero-length file %s - skipping search",
                      path)
            return

        try:
            with gzip.open(path, 'rb') as fd:
                try:
                    # test if file is gzip
                    fd.read(1)
                    fd.seek(0)
                    return self._run_search(fd)
                except OSError:
                    pass

            with open(path, 'rb') as fd:
                return self._run_search(fd)
        except UnicodeDecodeError:
            log.exception("")
            # ignore the file if it can't be decoded
            log.debug("caught UnicodeDecodeError for path %s - skipping",
                      path)
        except EOFError as e:
            log.exception("")
            msg = ("an exception occurred while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e
        except Exception as e:
            log.exception("")
            msg = ("an unexpected exception occurred while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e

        return SearchTaskStats()


class SearchTaskStats(UserDict):

    def __init__(self):
        self.reset()

    def reset(self):
        self.data = {'lines_searched': 0,
                     'results': 0}

    def combine(self, stats):
        if not stats:
            return

        for key, val in stats.items():
            self.data[key] += val

    def __repr__(self):
        return str(self.data)


class SearcherBase(abc.ABC):

    @abc.abstractproperty
    def files(self):
        """ Returns a list of files we will be searching. """

    @abc.abstractproperty
    def num_parallel_tasks(self):
        """
        Returns an integer representing the maximum number of tasks we can
        run in parallel. This will typically be bound by the number of
        cpu threads available.
        """

    @abc.abstractmethod
    def add(self, searchdef):
        """
        Add a search criterea.

        @param searchdef: SearchDef object
        """

    @abc.abstractmethod
    def run(self):
        """
        Execute all searches.
        """


class SearchConstraintsManager(object):

    def __init__(self, search_catalog):
        self.search_catalog = search_catalog
        self.global_constraints = []
        self.global_restrictions = set()

    def apply_global(self, user_path, fd):
        """ Apply any global constraints to the entire file. """
        offset = 0
        if not self.global_constraints:
            log.debug("no global constraint to apply to %s", fd.name)
            return offset

        if user_path in self.global_restrictions:
            log.debug("skipping global constraint for %s", fd.name)
            return offset

        # if not os.path.isdir(os.path.join(HotSOSConfig.data_root,
        #                                   'sos_commands')):
        #     log.info("skipping global constraints since data_root is not a "
        #              "sosreport therefore files may be changing")
        #     return offset

        for c in self.global_constraints:
            log.debug("applying task global constraint %s to %s", c, fd.name)
            _offset = c.apply_to_file(fd)
            if _offset is not None:
                return _offset

        return offset

    def apply_single(self, searchdef, line):
        """
        Apply any constraints for this searchdef to the give line.
        """
        if not searchdef.constraints:
            return True

        for c in searchdef.constraints.values():
            if c.apply_to_line(line):
                continue

            return False

        return True


class FileSearcher(SearcherBase):

    def __init__(self, max_parallel_tasks=8, max_logrotate_depth=7,
                 constraint=None):
        """
        @param max_parallel_tasks: max number of search tasks that can run in
                                   parallel.
        @param max_logrotate_depth: TODO
        @param constraint: constraint to be used with this
                                   searcher that applies to all files searched.
        """
        self.max_parallel_tasks = max_parallel_tasks
        self._stats = SearchTaskStats()
        self.catalog = SearchCatalog(max_logrotate_depth)
        self.constraints_manager = SearchConstraintsManager(self.catalog)
        if constraint:
            self.constraints_manager.global_constraints.append(constraint)

    @property
    def files(self):
        return [e['path'] for e in self.catalog]

    def resolve_source_id(self, source_id):
        return self.catalog.source_id_to_path(source_id)

    def add(self, searchdef, path, allow_global_constraints=True):
        """
        Add a search definition.

        @param searchdef: a SearchDef object
        @param path: path we want to search. this can be a file, dir or glob
        @param allow_global_constraints:
        """
        if not allow_global_constraints:
            self.constraints_manager.global_restrictions.add(path)

        self.catalog.register(searchdef, path)

    @property
    def num_parallel_tasks(self):
        if self.max_parallel_tasks == 0:
            cpus = 1  # i.e. no parallelism
        else:
            cpus = min(self.max_parallel_tasks, os.cpu_count())

        return min(len(self.files) or 1, cpus)

    @property
    def stats(self):
        """
        Provide stats for the last search run.

        @return: SearchTaskStats object
        """
        return self._stats

    def _get_results(self, results, queue_id, expected=None, event=None):
        _queue = RESULTS_QUEUES[queue_id]
        while True:
            if not _queue.empty():
                results.add(_queue.get())
            elif expected and expected > len(results):
                try:
                    r = _queue.get(timeout=RESULTS_QUEUE_TIMEOUT)
                    results.add(r)
                except queue.Empty:
                    log.debug("timeout waiting > %s secs to receive results - "
                              "expected=%s, actual=%s", RESULTS_QUEUE_TIMEOUT,
                              expected, len(results))
            elif event:
                if event.is_set():
                    log.debug("exiting get thread")
                    break
            else:
                break

    def _run(self, queue_id):
        """ Run all searches.

        @return: SearchResultsCollection object
        """
        self.stats.reset()
        results = SearchResultsCollection(self.catalog)
        jobs = []
        with multiprocessing.Manager() as m:
            lock = m.Lock()
            event = threading.Event()
            event.clear()
            t = threading.Thread(target=self._get_results,
                                 args=[results, queue_id],
                                 kwargs={'event': event})
            log.debug("starting queue consumer thread")
            t.start()
            with multiprocessing.Pool(processes=self.num_parallel_tasks) as p:
                for info in self.catalog:
                    c_mgr = self.constraints_manager
                    task = SearchTask(info, lock, constraints_manager=c_mgr,
                                      results_queue_id=queue_id)
                    jobs.append(p.apply_async(task.execute,
                                              error_callback=task.failed))

                log.debug("filesearcher: syncing %s jobs", len(jobs))
                while len(jobs) > 0:
                    for i, j in enumerate(jobs):
                        j.wait(10)
                        if j.ready():
                            self.stats.combine(j.get())
                            jobs.pop(i)

                log.debug("joining queue (expected=%s, remaining=%s)",
                          self.stats['results'],
                          self.stats['results'] - len(results))
                log.debug("joining queue consumer thread")
                event.set()
                t.join()
                self._get_results(results, queue_id,
                                  expected=self.stats['results'])

        log.debug("filesearcher: stats=%s", self.stats)
        log.debug("filesearcher: completed (results=%s)", len(results))
        return results

    def run(self):
        """ Run all searches.

        @return: SearchResultsCollection object
        """
        queue_id = "{}:{}".format(os.getpid(), uuid.uuid4())
        if queue_id in RESULTS_QUEUES:
            raise Exception("results queue already exists with id {}".
                            format(queue_id))

        RESULTS_QUEUES[queue_id] = multiprocessing.Queue()
        try:
            return self._run(queue_id)
        finally:
            del RESULTS_QUEUES[queue_id]
