# SPDX-FileCopyrightText: © 2020-2022 Jesse Johnson <jpj6652@gmail.com>
# SPDX-License-Identifier: LGPL-3.0-or-later
"""Parse git commit messages."""

# import logging
import os
from typing import TYPE_CHECKING, Any, Dict, List, Optional

from proman.common.exception import PromanException
from pygit2 import (
    GIT_OBJ_COMMIT,
    Config,
    Commit,
    RemoteCallbacks,
    Signature,
    Tag,
    UserPass,
)

from . import templates

if TYPE_CHECKING:
    from pygit2 import Repository


class Git:
    """Provide settings for git repositories."""

    def __init__(self, repo: 'Repository') -> None:
        """Initialize git object."""
        self.repo = repo
        self.hooks_dir = os.path.join(self.repo.path, 'hooks')
        self.config = os.path.join(self.repo.path, 'config')

    @property
    def username(self) -> str:
        """Return username from git configuration."""
        usernames = list(self.repo.config.get_multivar('user.name'))
        if usernames == []:
            raise PromanException('git user.name not configured')
        return usernames[-1]

    @property
    def email(self) -> str:
        """Return email from git configuration."""
        emails = list(self.repo.config.get_multivar('user.email'))
        if emails == []:
            raise PromanException('git user.email not configured')
        return emails[-1]

    @property
    def repo_dir(self) -> str:
        """Return directory of repository."""
        return self.repo.path

    @property
    def working_dir(self) -> str:
        """Return working directory of project."""
        return os.path.abspath(os.path.join(self.repo_dir, os.pardir))

    @property
    def ref(self) -> str:
        """Retrieve the current reference."""
        return self.repo.head.name

    def add_remote(
        self,
        name: str,
        url: str,
        fetch: Optional[str] = None,
        **kwargs: Any,
    ) -> None:
        """Add remote."""
        if name not in [x.name for x in self.repo.remotes]:
            if fetch is None:
                fetch = f"+refs/heads/*:refs/remotes/{name}/*"
            self.repo.remotes.create(name, url, fetch)
        elif self.repo.remotes[name].url != url:
            self.repo.remotes.set_url(name, url)

    def commit(
        self,
        branch: Optional[str] = None,
        filepaths: List[str] = [],
        **kwargs: Any,
    ) -> Optional[Commit]:
        """Create commit."""
        ref = f"refs/heads/{branch}" if branch else self.ref

        author = Signature(self.username, self.email)
        committer = kwargs.get('commiter', author)
        message = kwargs.get('message', 'ci: bump version')

        # populate index
        index = self.repo.index
        if filepaths == []:
            index.add_all()
        else:
            for filepath in filepaths:
                index.add(
                    os.path.relpath(
                        filepath, os.path.join(self.repo.path, os.pardir)
                    )
                )
        index.write()
        tree = index.write_tree()

        # get parent
        parents = kwargs.get('parents', [self.repo.head.target])
        encoding = kwargs.get('encoding', 'utf-8')

        # commit
        if not kwargs.get('dry_run', False):
            commit = self.repo.create_commit(
                ref,
                author,
                committer,
                message,
                tree,
                parents,
                encoding,
            )
            return commit
        # TODO: should a mock commit be created?
        return None

    def tag(
        self, name: str, branch: Optional[str] = None, **kwargs: Any
    ) -> Optional[Tag]:
        """Create tag."""
        ref = f"refs/heads/{branch}" if branch else self.ref
        commit = self.repo.resolve_refish(ref)[0]
        oid = commit.hex
        kind = kwargs.get('kind', GIT_OBJ_COMMIT)

        # tagger = pygit2.Signature('Alice Doe', 'adoe@example.com', 12347, 0)
        tagger = Signature(self.username, self.email)
        message = kwargs.get('message')
        if not kwargs.get('dry_run', False):
            tag = self.repo.create_tag(
                name, oid, kind, tagger, message or f"ci: {name}"
            )
            return tag
        # TODO: should a mock tag be created?
        return None

    def push(
        self,
        branch: Optional[str] = None,
        remote: str = 'origin',
        remote_branch: Optional[str] = None,
        username: Optional[str] = None,
        password: Optional[str] = None,
    ) -> None:
        """Push commited changes."""
        # TODO: feels like remote should be encapsulated with creds
        remote_branch = remote_branch or branch or self.ref.split('/')[-1]
        # ref = f"refs/heads/{branch}" if branch else self.ref
        # remote_ref = f"refs/remotes/{remote}/{remote_branch}"
        # print(ref, remote_ref)

        remote_repo = self.repo.remotes[remote]
        credentials = UserPass(username, password)
        remote_repo.credentials = credentials

        callbacks = RemoteCallbacks(credentials=credentials)
        remote_repo.push([f"refs/heads/{branch}"], callbacks=callbacks)

    @staticmethod
    def _to_dict(config: Config) -> Dict[str, Any]:
        """Convert gitconfig to dictionary."""
        sections: Dict[str, Any] = {}
        for entry in config:
            s = sections
            if hasattr(entry, 'name'):
                items = entry.name.split('.')
                for index, part in enumerate(items):
                    if index < (len(items) - 1):
                        s = s.setdefault(part, {})
                    else:
                        s[part] = entry.value
        return sections

    @staticmethod
    def dump(
        data: Dict[str, Any],
        dest: str,
    ) -> None:
        """Create git hook."""
        content = templates.render(
            data={'sections': data}, template_name='gitconfig'
        )
        with open(dest, 'w+') as f:
            f.write(content)

    def load(
        self,
        filepath: Optional[str] = None,
        scope: Optional[str] = None,
    ) -> Dict[str, Any]:
        """Get git config as dictionary."""
        if filepath:
            cfg = Config(filepath)
        elif scope == 'system':
            cfg = self.repo.config.get_system_config()
        elif scope == 'global':
            cfg = self.repo.config.get_global_config()
        elif scope == 'xdg':
            cfg = self.repo.config.get_xdg_config()
        else:
            cfg = Config(os.path.join(self.repo.path, 'config'))
        return self._to_dict(cfg)
