"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatInstanceImage = exports.NatInstanceProvider = exports.NatProvider = exports.NatTrafficDirection = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * Direction of traffic to allow all by default.
 *
 * @stability stable
 */
var NatTrafficDirection;
(function (NatTrafficDirection) {
    NatTrafficDirection["OUTBOUND_ONLY"] = "OUTBOUND_ONLY";
    NatTrafficDirection["INBOUND_AND_OUTBOUND"] = "INBOUND_AND_OUTBOUND";
    NatTrafficDirection["NONE"] = "NONE";
})(NatTrafficDirection = exports.NatTrafficDirection || (exports.NatTrafficDirection = {}));
/**
 * NAT providers.
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @stability stable
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC.
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     * @stability stable
     */
    static gateway(props = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatGatewayProps(props);
        return new NatGatewayProvider(props);
    }
    /**
     * Use NAT instances to provide NAT services for your VPC.
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     * @stability stable
     */
    static instance(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatInstanceProps(props);
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
_a = JSII_RTTI_SYMBOL_1;
NatProvider[_a] = { fqn: "@aws-cdk/aws-ec2.NatProvider", version: "1.144.0" };
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor(props = {}) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        if (this.props.eipAllocationIds != null
            && !core_1.Token.isUnresolved(this.props.eipAllocationIds)
            && this.props.eipAllocationIds.length < options.natSubnets.length) {
            throw new Error(`Not enough NAT gateway EIP allocation IDs (${this.props.eipAllocationIds.length} provided) for the requested subnet count (${options.natSubnets.length} needed).`);
        }
        // Create the NAT gateways
        let i = 0;
        for (const sub of options.natSubnets) {
            const eipAllocationId = this.props.eipAllocationIds ? pickN(i, this.props.eipAllocationIds) : undefined;
            const gateway = sub.addNatGateway(eipAllocationId);
            this.gateways.add(sub.availabilityZone, gateway.ref);
            i++;
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * NAT provider which uses NAT Instances.
 *
 * @stability stable
 */
class NatInstanceProvider extends NatProvider {
    /**
     * @stability stable
     */
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatInstanceProps(props);
        if (props.defaultAllowedTraffic !== undefined && props.allowAllTraffic !== undefined) {
            throw new Error('Can not specify both of \'defaultAllowedTraffic\' and \'defaultAllowedTraffic\'; prefer \'defaultAllowedTraffic\'');
        }
    }
    /**
     * Called by the VPC to configure NAT.
     *
     * Don't call this directly, the VPC will call it automatically.
     *
     * @stability stable
     */
    configureNat(options) {
        var _d, _e, _f;
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ConfigureNatOptions(options);
        const defaultDirection = (_d = this.props.defaultAllowedTraffic) !== null && _d !== void 0 ? _d : (((_e = this.props.allowAllTraffic) !== null && _e !== void 0 ? _e : true) ? NatTrafficDirection.INBOUND_AND_OUTBOUND : NatTrafficDirection.OUTBOUND_ONLY);
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_f = this.props.securityGroup) !== null && _f !== void 0 ? _f : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
            allowAllOutbound: isOutboundAllowed(defaultDirection),
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if (isInboundAllowed(defaultDirection)) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * The Security Group associated with the NAT instances.
     *
     * @stability stable
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * Manage the Security Groups associated with the NAT instances.
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    /**
     * Return list of gateways spawned by the provider.
     *
     * @stability stable
     */
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    /**
     * Configures subnet with the gateway.
     *
     * Don't call this directly, the VPC will call it automatically.
     *
     * @stability stable
     */
    configureSubnet(subnet) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnet(subnet);
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
_b = JSII_RTTI_SYMBOL_1;
NatInstanceProvider[_b] = { fqn: "@aws-cdk/aws-ec2.NatInstanceProvider", version: "1.144.0" };
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image.
 *
 * @stability stable
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    /**
     * @stability stable
     */
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
_c = JSII_RTTI_SYMBOL_1;
NatInstanceImage[_c] = { fqn: "@aws-cdk/aws-ec2.NatInstanceImage", version: "1.144.0" };
function isOutboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND ||
        direction === NatTrafficDirection.OUTBOUND_ONLY;
}
function isInboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND;
}
/**
 * Token-aware pick index function
 */
function pickN(i, xs) {
    if (core_1.Token.isUnresolved(xs)) {
        return core_1.Fn.select(i, xs);
    }
    if (i >= xs.length) {
        throw new Error(`Cannot get element ${i} from ${xs}`);
    }
    return xs[i];
}
//# sourceMappingURL=data:application/json;base64,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