# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyorthanc', 'pyorthanc.deprecated']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.23.0,<0.24.0', 'pydicom>=2.3.0,<3.0.0']

setup_kwargs = {
    'name': 'pyorthanc',
    'version': '1.11.3',
    'description': 'Orthanc REST API python wrapper with additional utilities',
    'long_description': "PyOrthanc\n=========\nPython library that wraps the Orthanc REST API and facilitates the manipulation\nof data with several cool utilities.\n\n\nBreaking changes\n----------------\nPyOrthanc has been rewritten almost entirely from `0.*.*`.\nThe new `Orthanc` client is now automatically generated from `https://api.orthanc-server.com/`.\nThe version `*.*.` of PyOrthanc will follow Orthanc version \n(e.g. `pyorthanc 1.11.*` is generated from the API specification of Orthanc `1.11.*`).\n\nThis means that the method names of the `Orthanc` objects from `PyOrthanc 0.*` are no longer the same.\nYou can still use the old client with\n```python\nfrom pyorthanc.deprecated.client import Orthanc\n```\n\nNote that due to automatic generation some method names may be less clear.\nHowever, the automatic generation allows PyOrthanc to cover all the routes of the API of Orthanc.\n\n\n\nInstallation\n------------\n```sh\n$ pip install pyorthanc\n```\n\nExample of usage\n----------------\nBe sure that Orthanc is running. The default URL (if running locally) is `http://localhost:8042`.\n\n#### Getting access to patients, studies, series and instances information:\n\n```python\nfrom pyorthanc import Orthanc\n\northanc = Orthanc('http://localhost:8042', username='username', password='password')\n\n# To get patients identifier and main information\npatients_identifiers = orthanc.get_patients()\n\nfor patient_identifier in patients_identifiers:\n   # To get patient information\n   patient_info = orthanc.get_patients_id(patient_identifier)\n\n   patient_name = patient_info['MainDicomTags']['PatientName']\n   ...\n   study_identifiers = patient_info['Studies']\n\n# To get patient's studies identifier and main information\nfor study_identifier in study_identifiers:\n   # To get Study info\n   study_info = orthanc.get_studies_id(study_identifier)\n\n   study_date = study_info['MainDicomTags']['StudyDate']\n   ...\n   series_identifiers = study_info['Series']\n\n# To get study's series identifier and main information\nfor series_identifier in series_identifiers:\n   # Get series info\n   series_info = orthanc.get_series_id(series_identifier)\n\n   modality = series_info['MainDicomTags']['Modality']\n   ...\n   instance_identifiers = series_info['Instances']\n\n# and so on ...\nfor instance_identifier in instance_identifiers:\n   instance_info = orthanc.get_instances_id(instance_identifier)\n   ...\n```\n\n#### Find patients with certain characteristics in an Orthanc instance:\nEach patient is a tree. Layers in each tree have the following structure \n`Patient` -> `Study` -> `Series` -> `Instance`\nthat correspond to the provided filter functions.\n\n```python\nfrom pyorthanc import find\n\npatients = find(\n    orthanc_url='http://localhost:8042/',\n    auth=('username', 'password'),\n    series_filter=lambda s: s.modality == 'RTDOSE'  # Optional: filter with pyorthanc.Series object\n)\n\nfor patient in patients:\n   patient_info = patient.get_main_information()\n   patient.id_   # Access PatientID\n   patient.name  # Access PatientName\n   \n   patient.get_zip() # DICOM files' content in bytes\n   \n   anonymized_patient_1 = patient.anonymize()  # New patient that was anonymized by Orthanc\n   anonymized_patient_2 = patient.anonymize(\n      keep=['PatientName'],   # You can keep/remove/replace the DICOM tags you want\n      replace={'PatientID': 'TheNewPatientID'},\n      remove=['ReferringPhysicianName'],\n      force=True  # Needed when changing PatientID/StudyInstanceUID/SeriesInstanceUID/SOPInstanceUID\n   )  \n   ...\n\n   for study in patient.studies:\n      study.date  # Date as a datetime object\n      study.referring_physician_name\n      ...\n\n      for series in study.series:\n         series.modality  # Should be 'RTDOSE' because of the series_filter parameters\n         ...\n```\n\n\n#### Upload DICOM files to Orthanc:\n\n```python\nfrom pyorthanc import Orthanc\n\northanc = Orthanc('http://localhost:8042', 'username', 'password')\n\nwith open('A_DICOM_INSTANCE_PATH.dcm', 'rb') as file:\n   orthanc.post_instances(file.read())\n```\n\n#### Getting list of connected remote modalities:\n```python\nfrom pyorthanc import Orthanc\n\northanc = Orthanc('http://localhost:8042', 'username', 'password')\n\northanc.get_modalities()\n```\n\n#### Query (C-Find) and Retrieve (C-Move) from remote modality:\n\n```python\nfrom pyorthanc import RemoteModality, Orthanc\n\northanc = Orthanc('http://localhost:8042', 'username', 'password')\n\nmodality = RemoteModality(orthanc, 'modality')\n\n# Query (C-Find) on modality\ndata = {'Level': 'Study', 'Query': {'PatientID': '*'}}\nquery_response = modality.query(data=data)\n\nanswer = modality.get_query_answers()[query_response['ID']]\nprint(answer)\n\n# Retrieve (C-Move) results of query on a target modality (AET)\nmodality.move(query_response['ID'], {'TargetAet': 'target_modality'})\n```\n\n#### Anonymize patient:\n```python\nfrom pyorthanc import Orthanc, Patient\n\northanc = Orthanc('http://localhost:8042', 'username', 'password')\n\npatient_identifier = orthanc.get_patients()[0]\n\nanonymized_patient = Patient(patient_identifier, orthanc).anonymize(\n    keep=['PatientName'],   # You can keep/remove/replace the DICOM tags you want\n    replace={'PatientID': 'TheNewPatientID'},\n    remove=['ReferringPhysicianName'],\n    force=True  # Needed when changing PatientID/StudyInstanceUID/SeriesInstanceUID/SOPInstanceUID\n)\n# Or directly with\northanc.post_patients_id_anonymize(patient_identifier)\n\n# result is: (you can retrieve DICOM file from ID)\n# {'ID': 'dd41f2f1-24838e1e-f01746fc-9715072f-189eb0a2',\n#  'Path': '/patients/dd41f2f1-24838e1e-f01746fc-9715072f-189eb0a2',\n#  'PatientID': 'dd41f2f1-24838e1e-f01746fc-9715072f-189eb0a2',\n#  'Type': 'Patient'}\n```\n\n## Citation\nIf you publish using PyOrthanc, we kindly ask that you credit us. PyOrthanc can be found on Zenodo :\nhttps://zenodo.org/record/7086219 .\n\n\n## Contributing\nYou can contribute to this project with the following steps:\n1. First, fork the project on Github \n2. Clone the project\n   ```shell\n   git clone https://github.com/<your-github-username>/pyorthanc\n   cd pyorthanc\n   ```\n3. Enter the project and create a poetry environment \n   (this project use the [poetry](https://python-poetry.org/) for dependency management)\n   ```shell\n   peotry install \n   ```\n4. Make a new git branch where you will apply the changes\n   ```shell\n   git checkout -b your-branch-name\n   ```\n   Now you can make your changes\n5. Once done, `git add`, `git commit` and `git push` the changes.\n6. Make a Pull Request from your branch to the https://github.com/ulaval-rs/pyorthanc.\n",
    'author': 'Gabriel Couture',
    'author_email': 'gacou54@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
