"use strict";
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
class ReusableStateMachine extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const choice = new stepfunctions.Choice(this, 'Choice')
            .when(stepfunctions.Condition.stringEquals('$.branch', 'left'), new stepfunctions.Pass(this, 'Left Branch'))
            .when(stepfunctions.Condition.stringEquals('$.branch', 'right'), new stepfunctions.Pass(this, 'Right Branch'));
        this.startState = choice;
        this.endStates = choice.afterwards().endStates;
    }
}
class SimpleChain extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const task1 = new stepfunctions.Task(this, 'Task1', { task: new FakeTask() });
        this.task2 = new stepfunctions.Task(this, 'Task2', { task: new FakeTask() });
        task1.next(this.task2);
        this.startState = task1;
        this.endStates = [this.task2];
    }
    catch(state, props) {
        this.task2.addCatch(state, props);
        return this;
    }
}
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask {
    bind(_task) {
        return {
            resourceArn: 'resource'
        };
    }
}
module.exports = {
    'Basic composition': {
        'A single task is a State Machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Pass(stack, 'Some State');
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Some State',
                States: {
                    'Some State': { Type: 'Pass', End: true }
                }
            });
            test.done();
        },
        'A sequence of two tasks is a State Machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'You dont need to hold on to the state to render the entire state machine correctly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            task1.next(task2);
            // THEN
            test.deepEqual(render(task1), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A chain can be appended to'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Pass(stack, 'State Three');
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2)
                .next(task3);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A state machine can be appended to another state machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Wait(stack, 'State Three', {
                time: stepfunctions.WaitTime.duration(cdk.Duration.seconds(10))
            });
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(stepfunctions.Chain.start(task2).next(task3));
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Wait', End: true, Seconds: 10 },
                }
            });
            test.done();
        },
        'A state machine definition can be instantiated and chained'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const before = new stepfunctions.Pass(stack, 'Before');
            const after = new stepfunctions.Pass(stack, 'After');
            // WHEN
            const chain = before.next(new ReusableStateMachine(stack, 'Reusable')).next(after);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Before',
                States: {
                    'Before': { Type: 'Pass', Next: 'Choice' },
                    'Choice': {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.branch', StringEquals: 'left', Next: 'Left Branch' },
                            { Variable: '$.branch', StringEquals: 'right', Next: 'Right Branch' },
                        ]
                    },
                    'Left Branch': { Type: 'Pass', Next: 'After' },
                    'Right Branch': { Type: 'Pass', Next: 'After' },
                    'After': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A success state cannot be chained onto'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const succeed = new stepfunctions.Succeed(stack, 'Succeed');
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            test.throws(() => {
                pass.next(succeed).next(pass);
            });
            test.done();
        },
        'A failure state cannot be chained onto'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const fail = new stepfunctions.Fail(stack, 'Fail', { error: 'X', cause: 'Y' });
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            test.throws(() => {
                pass.next(fail).next(pass);
            });
            test.done();
        },
        'Parallels can contain direct states'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            const three = new stepfunctions.Pass(stack, 'Three');
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(one.next(two));
            para.branch(three);
            // THEN
            test.deepEqual(render(para), {
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'One',
                                States: {
                                    One: { Type: 'Pass', Next: 'Two' },
                                    Two: { Type: 'Pass', End: true },
                                }
                            },
                            {
                                StartAt: 'Three',
                                States: {
                                    Three: { Type: 'Pass', End: true }
                                }
                            }
                        ]
                    }
                }
            });
            test.done();
        },
        'Parallels can contain instantiated reusable definitions'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(new ReusableStateMachine(stack, 'Reusable1').prefixStates('Reusable1/'));
            para.branch(new ReusableStateMachine(stack, 'Reusable2').prefixStates('Reusable2/'));
            // THEN
            test.deepEqual(render(para), {
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Reusable1/Choice',
                                States: {
                                    'Reusable1/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable1/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable1/Right Branch' },
                                        ]
                                    },
                                    'Reusable1/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable1/Right Branch': { Type: 'Pass', End: true },
                                }
                            },
                            {
                                StartAt: 'Reusable2/Choice',
                                States: {
                                    'Reusable2/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable2/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable2/Right Branch' },
                                        ]
                                    },
                                    'Reusable2/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable2/Right Branch': { Type: 'Pass', End: true },
                                }
                            },
                        ]
                    }
                }
            });
            test.done();
        },
        'State Machine Fragments can be wrapped in a single state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const reusable = new SimpleChain(stack, 'Hello');
            const state = reusable.toSingleState();
            test.deepEqual(render(state), {
                StartAt: 'Hello',
                States: {
                    Hello: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Hello: Task1',
                                States: {
                                    'Hello: Task1': { Type: 'Task', Next: 'Hello: Task2', Resource: 'resource' },
                                    'Hello: Task2': { Type: 'Task', End: true, Resource: 'resource' },
                                }
                            }
                        ],
                    },
                }
            });
            test.done();
        },
        'Chaining onto branched failure state ignores failure state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const yes = new stepfunctions.Pass(stack, 'Yes');
            const no = new stepfunctions.Fail(stack, 'No', { error: 'Failure', cause: 'Wrong branch' });
            const enfin = new stepfunctions.Pass(stack, 'Finally');
            const choice = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), yes)
                .otherwise(no);
            // WHEN
            choice.afterwards().next(enfin);
            // THEN
            test.deepEqual(render(choice), {
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'No',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    No: { Type: 'Fail', Error: 'Failure', Cause: 'Wrong branch' },
                    Finally: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Can include OTHERWISE transition for Choice in afterwards()'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), new stepfunctions.Pass(stack, 'Yes'))
                .afterwards({ includeOtherwise: true })
                .next(new stepfunctions.Pass(stack, 'Finally'));
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'Finally',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    Finally: { Type: 'Pass', End: true },
                }
            });
            test.done();
        }
    },
    'Goto support': {
        'State machines can have unconstrainted gotos'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            // WHEN
            const chain = one.next(two).next(one);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'One',
                States: {
                    One: { Type: 'Pass', Next: 'Two' },
                    Two: { Type: 'Pass', Next: 'One' },
                }
            });
            test.done();
        },
    },
    'Catches': {
        'States can have error branches'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addCatch(failure);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        },
        'Retries and errors with a result path'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addRetry({ errors: ['HTTPError'], maxAttempts: 2 }).addCatch(failure, { resultPath: '$.some_error' }).next(failure);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Catch: [{ ErrorEquals: ['States.ALL'], Next: 'Failed', ResultPath: '$.some_error' }],
                        Retry: [{ ErrorEquals: ['HTTPError'], MaxAttempts: 2 }],
                        Next: 'Failed',
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        },
        'Can wrap chain and attach error handler'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.next(task2).toSingleState('Wrapped').addCatch(errorHandler);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Wrapped',
                States: {
                    Wrapped: {
                        Type: 'Parallel',
                        Branches: [
                            {
                                StartAt: 'Task1',
                                States: {
                                    Task1: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        Next: 'Task2',
                                    },
                                    Task2: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        End: true,
                                    },
                                }
                            }
                        ],
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                        End: true
                    },
                    ErrorHandler: { Type: 'Pass', End: true }
                },
            });
            test.done();
        },
        'Chaining does not chain onto error handler state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler).next(task2);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ]
                    },
                    Task2: { Type: 'Task', Resource: 'resource', End: true },
                    ErrorHandler: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Chaining does not chain onto error handler, extended'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const task3 = new stepfunctions.Task(stack, 'Task3', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler)
                .next(task2.addCatch(errorHandler))
                .next(task3.addCatch(errorHandler));
            // THEN
            const sharedTaskProps = { Type: 'Task', Resource: 'resource', Catch: [{ ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' }] };
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: { Next: 'Task2', ...sharedTaskProps },
                    Task2: { Next: 'Task3', ...sharedTaskProps },
                    Task3: { End: true, ...sharedTaskProps },
                    ErrorHandler: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Error handler with a fragment'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            task1.addCatch(errorHandler)
                .next(new SimpleChain(stack, 'Chain').catch(errorHandler))
                .next(task2.addCatch(errorHandler));
            test.done();
        },
        'Can merge state machines with shared states'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            task1.addCatch(failure);
            task2.addCatch(failure);
            task1.next(task2);
            // THEN
            test.deepEqual(render(task1), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Task2: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        }
    },
    'State machine validation': {
        'No duplicate state IDs'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            state1.next(state2);
            // WHEN
            test.throws(() => {
                render(state1);
            });
            test.done();
        },
        'No duplicate state IDs even across Parallel branches'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            const parallel = new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1)
                .branch(state2);
            // WHEN
            test.throws(() => {
                render(parallel);
            });
            test.done();
        },
        'No cross-parallel jumps'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const state1 = new stepfunctions.Pass(stack, 'State1');
            const state2 = new stepfunctions.Pass(stack, 'State2');
            test.throws(() => {
                new stepfunctions.Parallel(stack, 'Parallel')
                    .branch(state1.next(state2))
                    .branch(state2);
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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