# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/aggregate.ipynb.

# %% ../notebooks/aggregate.ipynb 3
from __future__ import annotations  # To use class name in herself for typing

import glob
import json
import re
import time
import unittest
from dataclasses import dataclass, field
from enum import Enum, auto
from pathlib import Path
from typing import Dict, List, Union

import numpy as np
import pandas as pd
import requests
import ruamel.yaml
from dataclasses_json import dataclass_json
from ruamel.yaml.comments import CommentedSeq
from strenum import LowercaseStrEnum
from tqdm import tqdm

tc = unittest.TestCase()
# from leximpact_socio_fisca_simu_etat.logger import logger

default_extraction_date = time.strftime("%Y-%m-%d")
yaml = ruamel.yaml.YAML()

# %% auto 0
__all__ = [
    "tc",
    "default_extraction_date",
    "yaml",
    "r",
    "openfisca_variables",
    "DataStructure",
    "FileFormat",
    "Perimeter",
    "Reference",
    "Data",
    "LabelKey",
    "Aggregate",
    "AggregateManager",
]

# %% ../notebooks/aggregate.ipynb 5
# Get the list of variables available in OpenFisca-France
# It is used to test if an aggregate variable exist in OpenFisca
r = requests.get(
    url="https://git.leximpact.dev/leximpact/leximpact-socio-fiscal-openfisca-json/-/raw/master/variables_summaries.json",
    timeout=2,
)
tc.assertEqual(r.status_code, 200)
openfisca_variables = r.json()

# %% ../notebooks/aggregate.ipynb 6
class DataStructure(LowercaseStrEnum):
    DICT = auto()
    LIST = auto()
    MIXED = auto()
    DISTRIBUTION_10 = auto()
    DISTRIBUTION_100 = auto()
    DISTRIBUTION_1000 = auto()
    DECILE = auto()
    CENTILE = auto()
    COPULAS_10 = auto()
    COPULAS_100 = auto()


class FileFormat(Enum):
    YAML = auto()
    JSON = auto()


@dataclass_json
@dataclass
class Perimeter:
    """Store a perimeter for a variable.

    Args:
        entity (str): Variable entity. Example : "individual".
        period (str): Date of data. Example: "2019"
        geographic (str): Geographic area. Exemple : "France entière".
    """

    entity: str = ""
    period: str = ""
    geographic: str = ""


@dataclass_json
@dataclass
class Reference:
    title: str
    href: str


@dataclass_json
@dataclass
class Data:
    reference: List[Reference] = field(default_factory=list)
    date: str = ""
    data_structure: DataStructure = str(DataStructure.LIST)
    extraction_date: str = default_extraction_date
    values: Union[List, Dict] = field(default_factory=Dict)


@dataclass_json
@dataclass
class LabelKey:
    key: str
    label: str


@dataclass_json
@dataclass
class Aggregate:
    description: str = ""
    short_name: str = ""
    openfisca_variable: str = ""
    source_variable: str = ""
    unit: str = ""
    perimeter: Perimeter = None
    label_of_keys: Dict[str, str] = None
    data: List[Data] = field(default_factory=list)

    def merge_with(self, agg: Aggregate):
        """Merge with another Aggregate.

        Args:
            agg (Aggregate): The Aggregate to merge with
        """
        if type(agg) is not Aggregate:
            raise TypeError(f"Param agg is not an Aggregate() but a {type(agg)}")
        keys = self.__dict__.keys()
        for name in keys:
            val = getattr(self, name)
            if type(val) is str and getattr(self, name) == "":
                setattr(self, name, getattr(agg, name))
            elif val is None:
                setattr(self, name, getattr(agg, name))
            elif name == "data":
                if type(agg.data) not in [list, CommentedSeq]:
                    # print(agg.data)
                    raise TypeError(f"agg.data is not a list but a {type(agg.data)}")
                for new_data in agg.data:
                    date = new_data.date
                    ds = new_data.data_structure
                    # We look into new aggregate if same data exist
                    for i, data in enumerate(self.data):
                        if data.date == date and data.data_structure == ds:
                            # print(" meme date et meme datastructure => On remplace")
                            self.data[i] = new_data
                            break
                    else:
                        # Don't exist, add it
                        # print("Don't exist, add it")
                        self.data.append(new_data)


class AggregateManager:
    def __init__(
        self, aggregate: Aggregate = Aggregate(), aggregates_path: str = "../aggregates"
    ):
        self.aggregate = aggregate
        self.aggregates_path = aggregates_path
        yaml = ruamel.yaml.YAML()
        # Register dataclasses to allow saving to YAML
        yaml.register_class(Perimeter)
        yaml.register_class(Reference)
        yaml.register_class(Data)
        yaml.register_class(Aggregate)
        self.__yaml = yaml

    def save_to_json(self, filename: str):
        """
        Save to JSON
        Args:
            filename (str): The path and filename to write
        """
        path = Path(filename)
        path.parent.mkdir(parents=True, exist_ok=True)
        # logger.debug(f"Writing to {path}")
        with open(path, mode="w") as file:
            file.write(self.aggregate.to_json(indent=2))

    def _save_to_yaml(self, filename: str):
        """Save to YAML and overwrite the existing file.

        Args:
            filename (str): The path and filename to write
        """
        path = Path(filename)
        path.parent.mkdir(parents=True, exist_ok=True)
        # print(f"DEBUG Writing to {path}")
        with open(path, mode="w") as file:
            ruamel.yaml.YAML().dump(self.aggregate, file)

    def save_to_yaml(self, filename: str, merge_if_exist: bool = True):
        """
        Save to YAML and merge if file exist
        Args:
            filename (str): The path and filename to write
        """
        path = Path(filename)
        if path.is_file() and merge_if_exist:
            # YAML already exist, merging...
            old_agg = AggregateManager(aggregates_path=self.aggregates_path)
            old_agg.load_from_yaml(path)
            self.aggregate.merge_with(old_agg.aggregate)
        self._save_to_yaml(filename)

    def load_from_yaml(self, filename: str):
        """
        Read a YAML file to an Aggregate
        Args:
            filename (str): The path and filename to read
        """
        with open(filename) as file:
            self.aggregate = ruamel.yaml.YAML().load(file)

    def list_aggregates(self):
        """
        Find all YAML files
        Returns:
            dict: dict of variable, with filenames and data types.
        """
        aggregates = {}
        files = glob.glob(self.aggregates_path + "/**/*.yaml", recursive=True)
        for file in files:
            p = Path(file)
            var_name = p.parts[-1][:-5]
            data_type = str(DataStructure.MIXED)
            for type_name, _ in DataStructure.__dict__["_value2member_map_"].items():
                if type_name in file:
                    data_type = type_name
            if aggregates.get(var_name):
                aggregates[var_name]["filenames"].append(file)
                aggregates[var_name]["types"].append(data_type)

            else:
                aggregates[var_name] = {
                    "filenames": [file],
                    "types": [data_type],
                }
        return aggregates

    def load_aggregate(
        self, dataset: str, variable: str, year: str = None, data_structure: str = None
    ):
        """
        Build the filepath from the argument and load the Aggregate
        Args:
            dataset (str): Name of the Dataset (POTE, INSEE,...)
            variable (str): Name of the varible to read
            year (str): Year of data
            data_structure (str): Type of data
        """
        filename = AggregateManager.get_path(
            self.aggregates_path,
            dataset,
            variable,
            FileFormat.YAML,
            year,
            data_structure,
        )
        # Path(self.aggregates_path) / dataset / year / (variable + ".yaml")
        self.load_from_yaml(filename)

    def get_aggregate_value(
        self,
        dataset: str = "POTE",
        var: str = "rfr",
        year: str = "2019",
        agg_type: str = "sum",
    ):
        """Return the value."""
        self.load_aggregate(dataset, var)
        for d in self.aggregate.data:
            if d.data_structure == "dict" and d.date == year:
                return d.values[agg_type]

    @staticmethod
    def get_path(
        aggregates_path: str,
        dataset: str,
        variable: str,
        file_format: FileFormat = FileFormat.YAML,
        year: str = None,
        data_structure: str = None,
    ):
        """
        Build the filepath from the argument
        Args:
            aggregates_path (str): The base path to look into
            dataset (str): Name of the Dataset (POTE, INSEE,...)
            variable (str): Name of the varible to read
            file_format (str): File extension, default: YAML
            year (str): Year of data
            data_structure (str): Type of data

        Returns:
            Path(): the Path to the aggregate
        """
        if file_format == FileFormat.YAML:
            ext = ".yaml"
        elif file_format == FileFormat.JSON:
            ext = ".json"
        if len(variable) < 1:
            raise ValueError("Variable name is empty !")
        if data_structure is not None and (
            data_structure.startswith("copulas") or data_structure.startswith("distrib")
        ):
            if year is None:
                raise ValueError(
                    f"AggregateManager.get_path : 'year' can't be 'None' when asking for data_structure type {data_structure}"
                )
            sub_folder = Path(data_structure) / Path(year)
        else:
            sub_folder = "mixed"
        return Path(aggregates_path) / dataset / sub_folder / (variable.lower() + ext)

    @staticmethod
    def save_agg_to_yaml_and_json(
        filepath: Path,
        aggregate: Aggregate,
        aggregates_path: str = "../aggregates",
    ):
        """
        Save an Aggregate into JSON and YAML
        Args:
            filepath (Path): The base path to save into
            aggregate (Aggregate): The Aggregate to save

        Returns:
            Nothing, the files are saved on disk
        """
        aggm = AggregateManager(aggregate, aggregates_path=aggregates_path)
        aggm.save_to_yaml(filepath)
        aggm.save_to_json(filepath.with_suffix(".json"))

    @staticmethod
    def save_agg_from_list(
        agg_list: List[Aggregate],
        dataset: str = "POTE",
        year: str = "DATE",
        aggregates_path: str = "../aggregates",
        data_structure: DataStructure = None,
        debug=False,
    ) -> None:
        """
        Save some Aggregate into JSON and YAML
        Args:
            agg_list (List[Aggregate]): List of aggregate
            dataset (str): The dataset name
            year (str): Year of data
            aggregates_path (Path): The base path to save into
            data_structure (DataStructure): Type of data

        Returns:
            Nothing, the files are saved on disk
        """
        # tc.assertEqual(type(data_structure), DataStructure)
        tc.assertNotEqual(year, "DATE")
        for agg in tqdm(agg_list):
            if agg.openfisca_variable == "":
                if openfisca_variables.get(agg.source_variable):
                    var_name = agg.source_variable
                    agg.openfisca_variable = var_name
                    agg.description = openfisca_variables[var_name].get("label", "")
                var_name = agg.source_variable
            else:
                var_name = agg.openfisca_variable
            if var_name == "":
                raise ValueError("source_variable and openfisca_variable are empty !")

            path = Path(aggregates_path) / dataset / year / (var_name + ".yaml")
            current_data_structure = data_structure
            if not current_data_structure:
                ds = agg.data[-1].data_structure
                if ds.startswith("copulas") or ds.startswith("distrib"):
                    current_data_structure = ds
                if ds == "quantile_10":
                    current_data_structure = DataStructure.DECILE
                    agg.data[-1].data_structure = str(current_data_structure)
                if ds == "quantile_100":
                    current_data_structure = DataStructure.CENTILE
                    agg.data[-1].data_structure = str(current_data_structure)
            if not current_data_structure:
                current_data_structure = DataStructure.MIXED
            path = AggregateManager.get_path(
                aggregates_path,
                dataset,
                var_name,
                FileFormat.YAML,
                year,
                current_data_structure,
            )
            if debug:
                print(f"Saving to {path}")
            AggregateManager.save_agg_to_yaml_and_json(path, agg, aggregates_path)

    @staticmethod
    def get_key_value_aggregate_from_csv(
        csv_filepath: str,
        var_name: str,
        metadata: dict,
        year: str,
        ref: Reference,
        perimeter: Perimeter,
    ) -> Aggregate:
        df = pd.read_csv(csv_filepath)
        short_name = description = label_of_keys = openfisca_variable = ""
        if metadata:
            openfisca_variable = metadata.get("openfisca_variable", "")
            short_name = metadata.get("short_name", openfisca_variable)
            description = metadata.get("description", "")
            label_of_keys = metadata.get("labels", "")
        values = {}
        _ = df.apply(AggregateManager.row_to_dict, args=[values], axis=1)

        data = Data(reference=[ref], date=year, values=values, data_structure="dict")
        aggregate = Aggregate(
            description=description,
            short_name=short_name,
            openfisca_variable=openfisca_variable,
            source_variable=var_name,
            unit="foyer",
            perimeter=perimeter,
            label_of_keys=label_of_keys,
            data=[data],
        )
        return aggregate

    @staticmethod
    def get_key_value_aggregates(
        dataset_folder: str,
        year: str,
        metadata: dict,
        ref: Reference,
        perimeter: Perimeter,
    ) -> List[Aggregate]:
        """Find all file corresponding to param dataset_folder and make
        aggregate from them.

        dataset_folder must contain glob wildcard like "/tmp/*.csv" or
        "/tmp/file[0-9].csv"
        """
        aggregates = []
        files = glob.glob(dataset_folder)
        for file in files:
            var_name = file.split("_")[-1].split(".")[0].lower()
            year = file.split("_")[-2]
            # print(file, var_name, year)
            aggregate = AggregateManager.get_key_value_aggregate_from_csv(
                file, var_name, metadata.get(var_name.lower()), year, ref, perimeter
            )
            aggregates.append(aggregate)
        return aggregates

    @staticmethod
    def get_aggregats_from_row(
        row,
        agregats,
        year,
        col_name,
        ref: Reference,
        metadata: dict,
        perimeter: Perimeter,
        unit: str = "foyer",
    ):
        """Make one aggregate from a row of dataframe DataFrame format:
        variable_name,agg1,agg2,... For example: ```csv variable_name,sum,mean
        rfr,5456455,8456 rmi,4566,454 ```

        ::row:: une ligne de dataframe
        ::year:: year of the data
        """
        values = {}
        for i, cell in enumerate(row):
            if i == 0:
                var_name = cell
            else:
                values[col_name[i]] = cell

        data = Data(reference=[ref], date=year, data_structure="dict", values=values)
        of_name = short_name = description = ""
        var_name = var_name.lower()
        if metadata.get(var_name):
            of_name = metadata.get(var_name).get("openfisca_variable", "")
            description = metadata.get(var_name).get("description", "")
            short_name = metadata.get(var_name).get("short_name", of_name)
        aggregate = Aggregate(
            source_variable=var_name,
            short_name=short_name,
            description=description,
            openfisca_variable=of_name,
            unit=unit,
            perimeter=perimeter,
            data=[data],
        )
        agregats.append(aggregate)

    @staticmethod
    def get_aggregate_from_df(df, variable_name, agg_type: str = "sum"):
        """Permet de récupérer un agrégat de type determiné dans le dataframe.

        ::df:: Dataframe à lire
        ::variable_name:: Nom de la variable
        """
        return df.loc[df["name"] == variable_name, agg_type].iloc[0]

    @staticmethod
    def get_aggregate_from_dataframe(
        df: pd.DataFrame,
        var_name: str,
        agregats: List[Aggregate],
        year: str,
        ref: Reference,
        metadata: dict,
        perimeter: Perimeter,
        unit: str = "foyer",
    ):
        """
        Make one aggregate from a Pandas Dataframe
        DataFrame format:
        data1,data2,data2,...
        For example:
        ```csv
        Secteur,sum,mean
        CZ,5456455,8456
        AZ,4566,454
        ```
        Args:
            - df: A full Pandas DataFrame
            - year: year of the data
        """
        values = json.loads(df.to_json(orient="records"))

        data = Data(reference=[ref], date=year, data_structure="dict", values=values)
        of_name = short_name = description = ""
        var_name = var_name.lower()
        if metadata.get(var_name):
            of_name = metadata.get(var_name).get("openfisca_variable", "")
            description = metadata.get(var_name).get("description", "")
            short_name = metadata.get(var_name).get("short_name", of_name)
        aggregate = Aggregate(
            source_variable=var_name,
            short_name=short_name,
            description=description,
            openfisca_variable=of_name,
            unit=unit,
            perimeter=perimeter,
            data=[data],
        )
        agregats.append(aggregate)

    @staticmethod
    def get_aggregats_from_json(
        var_name: str,
        calib_filepath: str,
        ref: Reference,
        perimeter: Perimeter,
        metadata: dict,
        year: str = "DATE",
        unit: str = "euros",
        data_structure="distribution",
    ) -> Aggregate:
        """Read a json to create an Aggregate.

        ::var_name:: Name of variable
        ::calib_filepath:: Path to json file
        """
        tc.assertNotEqual(year, "DATE")
        with open(calib_filepath) as myfile:
            json_content = json.loads(myfile.read())
        if type(json_content) is dict and json_content.get("copules"):
            json_content = json_content.get("copules")
        if type(json_content) is dict and json_content.get("buckets"):
            json_content = json_content.get("buckets")
        # keep_upper_bound_secret(json_content)
        data = Data(
            reference=[ref],
            date=year,
            data_structure=data_structure,
            values=json_content,
        )
        of_name = ""
        of_name = short_name = description = ""
        var_name = var_name.lower()
        if metadata.get(var_name):
            of_name = metadata.get(var_name).get("openfisca_variable", "")
            description = metadata.get(var_name).get("description", "")
            short_name = metadata.get(var_name).get("short_name", of_name)
        aggregate = Aggregate(
            source_variable=var_name,
            short_name=short_name,
            description=description,
            openfisca_variable=of_name,
            unit=unit,
            perimeter=perimeter,
            data=[data],
        )
        return aggregate

    @staticmethod
    def process_all_json(
        path: str,
        ref: Reference,
        perimeter: Perimeter,
        metadata: dict,
        data_structure="distribution",
        debug=False,
    ) -> List[Aggregate]:
        """Find all JSON on disk in path and load them in a list. `path` must
        contain glob wildcard like "/tmp/*.json" or "/tmp/file[0-9].json".

        Returns:
            List[Aggregate]: A list of Aggregate
        """
        tc.assertIsNotNone(path)
        aggregates = []
        files = glob.glob(path)
        if len(files) == 0:
            print(f"No file for {path} !")
            return []
        for file in tqdm(files):
            #    Parcours les fichier de calibration et enregistre les agrégats dans le dictionnaire.
            # Extract info from file path
            m = re.finditer(r"[_|-](\d*)[_|-](\d*)[_|-]", file)
            for group in m:
                tmp = group.group().replace("-", "_")
                year = tmp.split("_")[2]
                bucket_size = tmp.split("_")[1]
                var_name = file[group.end() :].split(".")[0]
            if debug:
                print(f"Processing {var_name} {bucket_size} from {file} for {year}...")
            aggregate = AggregateManager.get_aggregats_from_json(
                var_name,
                file,
                ref,
                perimeter,
                metadata,
                year=year,
                data_structure=f"{data_structure}_{bucket_size}",
            )
            # if bucket_size == "10" and var_name == "revkire":
            #     print(aggregate)
            aggregates.append(aggregate)
        return aggregates

    @staticmethod
    def row_to_dict(row, dest_dict: dict):
        """
        WARNING : Use only with 2 columns
        Args:
            row (row of DataFrame()): The row to get key and value
            dest_dict (dict): The :class:`FileStorage` instance to wrap

        Returns:
            row of DataFrame(): same row as input
        """
        # Force cast of float to avoid problems with ramel.YAML() dump
        key = row[0] if not isinstance(row[0], np.float64) else float(row[0])
        val = row[1] if not isinstance(row[1], np.float64) else float(row[1])
        val = "secret" if np.isnan(val) else val

        dest_dict[key] = val
        return row
