# Imports
import argparse as ap
import os
import sys
import time
import subprocess

# Main function
def main():
    # Parse command line arguments
    parser = ap.ArgumentParser(description="Automate COBOL documentation generation using Copilot.")
    parser.add_argument("-p","--Prompt",type=str,help="Prompt directory for Copilot to generate documentation.")
    parser.add_argument("-c","--Cobol",type=str,help="COBOL source code directory OR File to be documented.")
    parser.add_argument("-o","--Output",type=str,help="Output directory for generated documentation.")
    args = parser.parse_args()

    # Create variables for paths and validate them
    prompt_path = args.Prompt
    cobol_path = args.Cobol
    output_path = args.Output

    if not os.path.exists(prompt_path):
        print(f"Error: Prompt path '{prompt_path}' does not exist.")
        sys.exit(1)
    
    if not os.path.exists(cobol_path):
        print(f"Error: COBOL path '{cobol_path}' does not exist.")
        sys.exit(1)
    
    if output_path is None:
        output_path = os.path.join(os.getcwd(), "Cobol_Documentation_Output")
        print(f"No output path provided. Using default: '{output_path}'")
    
    if not os.path.exists(output_path):
        os.makedirs(output_path)
        print(f"Created output directory at '{output_path}'.")
    
    print("Validation steps completed successfully. Moving to documentation generation...")

    # PROCESS SINGLE FILE ELSE DIRECTORY
    if os.path.isfile(cobol_path):
        process_cobol_file(cobol_path, prompt_path, output_path)
    else:
        for root, _, files in os.walk(cobol_path):
            for file in files:
                if file.lower().endswith(".cbl") or file.lower().endswith(".cob"):
                    cobol_file_path = os.path.join(root, file)
                    process_cobol_file(cobol_file_path, prompt_path, output_path)
    print("Documentation generation completed.")

def process_cobol_file(cobol_file_path, prompt_path, output_path):
    print(f"Processing COBOL file: {cobol_file_path}")

    for prompt_file in os.listdir(prompt_path):
        time.sleep(5)  # Pause for 5 seconds between processing each prompt file
        # Read the prompt into a variable
        with open(os.path.join(prompt_path, prompt_file), "r") as pf:
            prompt_file_content = pf.read()

        # Generate the final prompt by appending the cobol file path
        final_prompt = f"{prompt_file_content}, using the context of the cobol file located at {cobol_file_path}."
        
        # take the output in stdout created by copilot and pipe it to a file in output_path
        output_file_name = f"{os.path.splitext(os.path.basename(cobol_file_path))[0]}_{os.path.splitext(prompt_file)[0]}_documentation.txt"
        output_file_path = os.path.join(output_path, output_file_name)

        # Generate command sequence
        ps_command = (
        "$cmd = Get-Command copilot -ErrorAction SilentlyContinue; "
        "if (-not $cmd) { Write-Error 'copilot not found in PATH'; exit 2 }; "
        "$exe = $cmd.Source; "
        f"& $exe -p {_ps_single_quote(final_prompt)}"
        )

        # Attempt to execute the command and capture output
        with open(output_file_path, "w") as output_file:
            proc = subprocess.run(
                ["powershell.exe", "-NoProfile", "-ExecutionPolicy", "Bypass", "-Command", ps_command],
                capture_output=True,
                text=True
            )
            output_file.write(proc.stdout)

def _ps_single_quote(s: str) -> str:
    """
    Quote a string for use as a PowerShell single-quoted literal.
    In PowerShell single quotes are literal; to include a single-quote inside,
    double it:  'abc''def'
    """
    return "'" + s.replace("'", "''") + "'"

if __name__ == "__main__":
    main()
