"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../lib");
module.exports = {
    'basic pipeline'(test) {
        const stack = new core_1.Stack();
        const repository = new codecommit.Repository(stack, 'MyRepo', {
            repositoryName: 'my-repo',
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source],
        });
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        pipeline.addStage({
            stageName: 'build',
            actions: [
                new cpactions.CodeBuildAction({
                    actionName: 'build',
                    input: sourceOutput,
                    project,
                }),
            ],
        });
        test.notDeepEqual(assert_1.SynthUtils.toCloudFormation(stack), {});
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'Tokens can be used as physical names of the Pipeline'(test) {
        const stack = new core_1.Stack(undefined, 'StackName');
        new codepipeline.Pipeline(stack, 'Pipeline', {
            pipelineName: core_1.Aws.STACK_NAME,
        });
        assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Name": {
                "Ref": "AWS::StackName",
            },
        }));
        test.done();
    },
    'pipeline with GitHub source with poll trigger'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.POLL
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                {
                    "Actions": [
                        {
                            "Configuration": {
                                "PollForSourceChanges": true
                            },
                            "Name": "GH"
                        }
                    ],
                    "Name": "Source"
                },
                {
                    "Actions": [
                        {
                            "Name": "Boo",
                        }
                    ],
                    "Name": "Two"
                }
            ]
        }));
        test.done();
    },
    'pipeline with GitHub source without triggers'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.NONE
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                {
                    "Actions": [
                        {
                            "Configuration": {
                                "PollForSourceChanges": false
                            },
                            "Name": "GH"
                        }
                    ],
                    "Name": "Source"
                },
                {
                    "Actions": [
                        {
                            "Name": "Boo",
                        }
                    ],
                    "Name": "Two"
                }
            ]
        }));
        test.done();
    },
    'github action uses ThirdParty owner'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar'
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Webhook'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "ArtifactStore": {
                "Location": {
                    "Ref": "PArtifactsBucket5E711C12"
                },
                "Type": "S3"
            },
            "RoleArn": {
                "Fn::GetAtt": [
                    "PRole07BDC907",
                    "Arn"
                ]
            },
            "Stages": [
                {
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Source",
                                "Owner": "ThirdParty",
                                "Provider": "GitHub",
                                "Version": "1"
                            },
                            "Configuration": {
                                "Owner": "foo",
                                "Repo": "bar",
                                "Branch": "branch",
                                "OAuthToken": {
                                    "Ref": "GitHubToken"
                                },
                                "PollForSourceChanges": false
                            },
                            "Name": "GH",
                            "OutputArtifacts": [
                                {
                                    "Name": "A"
                                }
                            ],
                            "RunOrder": 8
                        }
                    ],
                    "Name": "Source"
                },
                {
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Approval",
                                "Owner": "AWS",
                                "Provider": "Manual",
                                "Version": "1"
                            },
                            "Name": "Boo",
                            "RunOrder": 1
                        }
                    ],
                    "Name": "Two"
                }
            ]
        }));
        test.deepEqual([], core_1.ConstructNode.validate(p.node));
        test.done();
    },
    'onStateChange'(test) {
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const pipeline = new codepipeline.Pipeline(stack, 'PL');
        pipeline.addStage({
            stageName: 'S1',
            actions: [
                new cpactions.S3SourceAction({
                    actionName: 'A1',
                    output: new codepipeline.Artifact('Artifact'),
                    bucket: new s3.Bucket(stack, 'Bucket'),
                    bucketKey: 'Key'
                }),
            ],
        });
        pipeline.addStage({
            stageName: 'S2',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'A2' }),
            ],
        });
        pipeline.onStateChange('OnStateChange', {
            target: new targets.SnsTopic(topic),
            description: 'desc',
            eventPattern: {
                detail: {
                    state: ['FAILED']
                }
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "Description": "desc",
            "EventPattern": {
                "detail": {
                    "state": [
                        "FAILED"
                    ]
                },
                "detail-type": [
                    "CodePipeline Pipeline Execution State Change"
                ],
                "source": [
                    "aws.codepipeline"
                ],
                "resources": [
                    {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    "Ref": "AWS::Partition"
                                },
                                ":codepipeline:",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ":",
                                {
                                    "Ref": "AWS::AccountId"
                                },
                                ":",
                                {
                                    "Ref": "PLD5425AEA"
                                }
                            ]
                        ]
                    }
                ]
            },
            "State": "ENABLED",
            "Targets": [
                {
                    "Arn": {
                        "Ref": "TopicBFC7AF6E"
                    },
                    "Id": "Target0"
                }
            ]
        }));
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'PipelineProject': {
        'with a custom Project Name': {
            'sets the source and artifacts to CodePipeline'(test) {
                const stack = new core_1.Stack();
                new codebuild.PipelineProject(stack, 'MyProject', {
                    projectName: 'MyProject',
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                    "Name": "MyProject",
                    "Source": {
                        "Type": "CODEPIPELINE"
                    },
                    "Artifacts": {
                        "Type": "CODEPIPELINE"
                    },
                    "ServiceRole": {
                        "Fn::GetAtt": [
                            "MyProjectRole9BBE5233",
                            "Arn"
                        ]
                    },
                    "Environment": {
                        "Type": "LINUX_CONTAINER",
                        "PrivilegedMode": false,
                        "Image": "aws/codebuild/standard:1.0",
                        "ComputeType": "BUILD_GENERAL1_SMALL"
                    }
                }));
                test.done();
            }
        }
    },
    'Lambda PipelineInvokeAction can be used to invoke Lambda functions from a CodePipeline'(test) {
        const stack = new core_1.Stack();
        const lambdaFun = new lambda.Function(stack, 'Function', {
            code: new lambda.InlineCode('bla'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const source1Output = new codepipeline.Artifact('sourceArtifact1');
        const source1 = new cpactions.S3SourceAction({
            actionName: 'SourceAction1',
            bucketKey: 'some/key',
            output: source1Output,
            bucket,
        });
        const source2Output = new codepipeline.Artifact('sourceArtifact2');
        const source2 = new cpactions.S3SourceAction({
            actionName: 'SourceAction2',
            bucketKey: 'another/key',
            output: source2Output,
            bucket,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                source1,
                source2,
            ],
        });
        const lambdaAction = new cpactions.LambdaInvokeAction({
            actionName: 'InvokeAction',
            lambda: lambdaFun,
            inputs: [
                source2Output,
                source1Output,
            ],
            outputs: [
                new codepipeline.Artifact('lambdaOutput1'),
                new codepipeline.Artifact('lambdaOutput2'),
                new codepipeline.Artifact('lambdaOutput3'),
            ],
        });
        pipeline.addStage({
            stageName: 'Stage',
            actions: [lambdaAction],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "ArtifactStore": {
                "Location": {
                    "Ref": "PipelineArtifactsBucket22248F97"
                },
                "Type": "S3"
            },
            "RoleArn": {
                "Fn::GetAtt": [
                    "PipelineRoleD68726F7",
                    "Arn"
                ]
            },
            "Stages": [
                {
                    "Name": "Source",
                },
                {
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Invoke",
                                "Owner": "AWS",
                                "Provider": "Lambda",
                                "Version": "1"
                            },
                            "Configuration": {
                                "FunctionName": {
                                    "Ref": "Function76856677"
                                },
                            },
                            "InputArtifacts": [
                                { "Name": "sourceArtifact2" },
                                { "Name": "sourceArtifact1" },
                            ],
                            "Name": "InvokeAction",
                            "OutputArtifacts": [
                                { "Name": "lambdaOutput1" },
                                { "Name": "lambdaOutput2" },
                                { "Name": "lambdaOutput3" },
                            ],
                            "RunOrder": 1
                        }
                    ],
                    "Name": "Stage"
                }
            ]
        }));
        test.equal((lambdaAction.actionProperties.outputs || []).length, 3);
        assert_1.expect(stack, /* skip validation */ true).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "codepipeline:PutJobSuccessResult",
                            "codepipeline:PutJobFailureResult"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "FunctionServiceRoleDefaultPolicy2F49994A",
            "Roles": [
                {
                    "Ref": "FunctionServiceRole675BB04A"
                }
            ]
        }));
        test.done();
    },
    'cross-region Pipeline': {
        'generates the required Action & ArtifactStores properties in the template'(test) {
            const pipelineRegion = 'us-west-2';
            const pipelineAccount = '123';
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'TestStack', {
                env: {
                    region: pipelineRegion,
                    account: pipelineAccount,
                },
            });
            const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
                crossRegionReplicationBuckets: {
                    'us-west-1': s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'sfo-replication-bucket'),
                },
            });
            const sourceBucket = new s3.Bucket(stack, 'MyBucket');
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            const sourceAction = new cpactions.S3SourceAction({
                actionName: 'BucketSource',
                bucketKey: '/some/key',
                output: sourceOutput,
                bucket: sourceBucket,
            });
            pipeline.addStage({
                stageName: 'Stage1',
                actions: [sourceAction],
            });
            pipeline.addStage({
                stageName: 'Stage2',
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'Action1',
                        changeSetName: 'ChangeSet',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'SomeStack',
                        region: pipelineRegion,
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationCreateUpdateStackAction({
                        actionName: 'Action2',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'OtherStack',
                        region: 'us-east-1',
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action3',
                        changeSetName: 'ChangeSet',
                        stackName: 'SomeStack',
                        region: 'us-west-1',
                    }),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "ArtifactStores": [
                    {
                        "Region": "us-west-1",
                        "ArtifactStore": {
                            "Type": "S3",
                            "Location": "sfo-replication-bucket",
                        },
                    },
                    {
                        "Region": "us-east-1",
                        "ArtifactStore": {
                            "Type": "S3",
                            "Location": "teststack-support-us-easteplicationbucket1a8063b3cdac6e7e0e73",
                        },
                    },
                    {
                        "Region": "us-west-2",
                        "ArtifactStore": {
                            "Type": "S3",
                            "EncryptionKey": {
                                "Type": "KMS",
                                "Id": {},
                            },
                        },
                    },
                ],
                "Stages": [
                    {
                        "Name": "Stage1",
                    },
                    {
                        "Name": "Stage2",
                        "Actions": [
                            {
                                "Name": "Action1",
                                "Region": "us-west-2",
                            },
                            {
                                "Name": "Action2",
                                "Region": "us-east-1",
                            },
                            {
                                "Name": "Action3",
                                "Region": "us-west-1",
                            },
                        ],
                    },
                ],
            }));
            test.notEqual(pipeline.crossRegionSupport[pipelineRegion], undefined);
            test.notEqual(pipeline.crossRegionSupport['us-west-1'], undefined);
            const usEast1Support = pipeline.crossRegionSupport['us-east-1'];
            test.notEqual(usEast1Support, undefined);
            test.equal(usEast1Support.stack.region, 'us-east-1');
            test.equal(usEast1Support.stack.account, pipelineAccount);
            test.ok(usEast1Support.stack.node.id.indexOf('us-east-1') !== -1, `expected '${usEast1Support.stack.node.id}' to contain 'us-east-1'`);
            test.done();
        },
        'allows specifying only one of artifactBucket and crossRegionReplicationBuckets'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new codepipeline.Pipeline(stack, 'Pipeline', {
                    artifactBucket: new s3.Bucket(stack, 'Bucket'),
                    crossRegionReplicationBuckets: {
                    // even an empty map should trigger this validation...
                    },
                });
            }, /Only one of artifactBucket and crossRegionReplicationBuckets can be specified!/);
            test.done();
        },
        'does not create a new artifact Bucket if one was provided in the cross-region Buckets for the Pipeline region'(test) {
            const pipelineRegion = 'us-west-2';
            const stack = new core_1.Stack(undefined, undefined, {
                env: {
                    region: pipelineRegion,
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                crossRegionReplicationBuckets: {
                    [pipelineRegion]: new s3.Bucket(stack, 'Bucket', {
                        bucketName: 'my-pipeline-bucket',
                    })
                },
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                repository: new codecommit.Repository(stack, 'Repo', { repositoryName: 'Repo' }),
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                project: new codebuild.PipelineProject(stack, 'Project'),
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::S3::Bucket', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "ArtifactStores": [
                    {
                        "Region": pipelineRegion,
                        "ArtifactStore": {
                            "Type": "S3",
                            "Location": {
                                "Ref": "Bucket83908E77",
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
        'allows providing a resource-backed action from a different region directly'(test) {
            const account = '123456789012';
            const app = new core_1.App();
            const replicationRegion = 'us-west-1';
            const replicationStack = new core_1.Stack(app, 'ReplicationStack', { env: { region: replicationRegion, account } });
            const project = new codebuild.PipelineProject(replicationStack, 'CodeBuildProject', {
                projectName: 'MyCodeBuildProject',
            });
            const pipelineRegion = 'us-west-2';
            const pipelineStack = new core_1.Stack(app, 'TestStack', { env: { region: pipelineRegion, account } });
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            new codepipeline.Pipeline(pipelineStack, 'MyPipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommitAction',
                                output: sourceOutput,
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'my-codecommit-repo'),
                            })],
                    },
                    {
                        stageName: 'Build',
                        actions: [new cpactions.CodeBuildAction({
                                actionName: 'CodeBuildAction',
                                input: sourceOutput,
                                project,
                            })],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "ArtifactStores": [
                    {
                        "Region": replicationRegion,
                        "ArtifactStore": {
                            "Type": "S3",
                            "Location": "replicationstackeplicationbucket2464cd5c33b386483b66",
                            "EncryptionKey": {
                                "Id": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":kms:us-west-1:123456789012:alias/ionstacktencryptionalias043cb2f8ceac9da9c07c",
                                        ],
                                    ],
                                },
                                "Type": "KMS"
                            },
                        },
                    },
                    {
                        "Region": pipelineRegion,
                    },
                ],
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuildAction",
                                "Region": replicationRegion,
                                "Configuration": {
                                    "ProjectName": "MyCodeBuildProject",
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::S3::Bucket', {
                "BucketName": "replicationstackeplicationbucket2464cd5c33b386483b66",
            }));
            test.done();
        },
    },
    'cross-account Pipeline': {
        'with a CodeBuild Project in a different account works correctly'(test) {
            const app = new core_1.App();
            const buildAccount = '901234567890';
            const buildRegion = 'bermuda-triangle-1';
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region: buildRegion },
            });
            const rolePhysicalName = 'ProjectRolePhysicalName';
            const projectRole = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
                roleName: rolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
                role: projectRole,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: '123456789012', region: buildRegion },
            });
            const sourceBucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket: sourceBucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuild",
                                "Configuration": {
                                    "ProjectName": projectPhysicalName,
                                },
                                "RoleArn": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            `:iam::${buildAccount}:role/buildstackebuildactionrole166c75d1d8be701b1ad8`,
                                        ],
                                    ],
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(buildStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                        // log permissions from the CodeBuild Project Construct...
                        },
                        {
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58',
                                        ],
                                    ],
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58/*',
                                        ],
                                    ],
                                },
                            ],
                        },
                        {
                            "Action": [
                                "kms:Decrypt",
                                "kms:DescribeKey",
                            ],
                            "Effect": "Allow",
                            "Resource": "*",
                        },
                    ],
                },
            }));
            test.done();
        },
        'adds a dependency on the Stack containing a new action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.AccountPrincipal(pipelineAccount),
                roleName: actionRolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = codebuild.Project.fromProjectName(buildStack, 'Project', projectPhysicalName);
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuild",
                                "Configuration": {
                                    "ProjectName": projectPhysicalName,
                                },
                                "RoleArn": {
                                    "Fn::Join": ["", [
                                            "arn:",
                                            { "Ref": "AWS::Partition" },
                                            `:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                                        ]],
                                },
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 1);
            test.done();
        },
        'does not add a dependency on the Stack containing an imported action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = iam.Role.fromRoleArn(buildStack, 'ProjectRole', `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`);
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuild",
                                "Configuration": {
                                    "ProjectName": projectPhysicalName,
                                },
                                "RoleArn": `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 0);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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