"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
module.exports = {
    'CodeBuild action': {
        'that is cross-account and has outputs': {
            'causes an error'(test) {
                const app = new core_1.App();
                const projectStack = new core_1.Stack(app, 'ProjectStack', {
                    env: {
                        region: 'us-west-2',
                        account: '012345678912',
                    },
                });
                const project = new codebuild.PipelineProject(projectStack, 'Project');
                const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                    env: {
                        region: 'us-west-2',
                        account: '012345678913',
                    },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new cpactions.CodeCommitSourceAction({
                                    actionName: 'CodeCommit',
                                    repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'repo-name'),
                                    output: sourceOutput,
                                })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({
                    stageName: 'Build',
                });
                // this works fine - no outputs!
                buildStage.addAction(new cpactions.CodeBuildAction({
                    actionName: 'Build1',
                    input: sourceOutput,
                    project,
                }));
                const buildAction2 = new cpactions.CodeBuildAction({
                    actionName: 'Build2',
                    input: sourceOutput,
                    project,
                    outputs: [new codepipeline.Artifact()],
                });
                test.throws(() => {
                    buildStage.addAction(buildAction2);
                }, /https:\/\/github\.com\/aws\/aws-cdk\/issues\/4169/);
                test.done();
            },
        },
        'can be backed by an imported project'(test) {
            const stack = new core_1.Stack();
            const codeBuildProject = codebuild.PipelineProject.fromProjectName(stack, 'CodeBuild', 'codeBuildProjectNameInAnotherAccount');
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: new s3.Bucket(stack, 'Bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                input: sourceOutput,
                                project: codeBuildProject,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuild",
                                "Configuration": {
                                    "ProjectName": "codeBuildProjectNameInAnotherAccount",
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const codeBuildAction = new cpactions.CodeBuildAction({
                actionName: 'CodeBuild',
                input: sourceOutput,
                project: new codebuild.PipelineProject(stack, 'CodeBuild', {
                    buildSpec: codebuild.BuildSpec.fromObject({
                        version: '0.2',
                        env: {
                            'exported-variables': [
                                'SomeVar',
                            ],
                        },
                        phases: {
                            build: {
                                commands: [
                                    'export SomeVar="Some Value"',
                                ],
                            },
                        },
                    }),
                }),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: s3.Bucket.fromBucketName(stack, 'Bucket', 'bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            codeBuildAction,
                            new cpactions.ManualApprovalAction({
                                actionName: 'Approve',
                                additionalInformation: codeBuildAction.variable('SomeVar'),
                                notificationTopic: sns.Topic.fromTopicArn(stack, 'Topic', 'arn:aws:sns:us-east-1:123456789012:mytopic'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "CodeBuild",
                                "Namespace": "Build_CodeBuild_NS",
                            },
                            {
                                "Name": "Approve",
                                "Configuration": {
                                    "CustomData": "#{Build_CodeBuild_NS.SomeVar}",
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb2RlYnVpbGQtYWN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5jb2RlYnVpbGQtYWN0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0Qsb0RBQW9EO0FBQ3BELHNEQUFzRDtBQUN0RCwwREFBMEQ7QUFDMUQsc0NBQXNDO0FBQ3RDLHdDQUF3QztBQUN4Qyx3Q0FBMkM7QUFFM0MsdUNBQXVDO0FBSXZDLGlCQUFTO0lBQ1Asa0JBQWtCLEVBQUU7UUFDbEIsdUNBQXVDLEVBQUU7WUFDdkMsaUJBQWlCLENBQUMsSUFBVTtnQkFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztnQkFFdEIsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtvQkFDbEQsR0FBRyxFQUFFO3dCQUNILE1BQU0sRUFBRSxXQUFXO3dCQUNuQixPQUFPLEVBQUUsY0FBYztxQkFDeEI7aUJBQ0YsQ0FBQyxDQUFDO2dCQUNILE1BQU0sT0FBTyxHQUFHLElBQUksU0FBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZLEVBQUUsU0FBUyxDQUFDLENBQUM7Z0JBRXZFLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLEVBQUU7b0JBQ3BELEdBQUcsRUFBRTt3QkFDSCxNQUFNLEVBQUUsV0FBVzt3QkFDbkIsT0FBTyxFQUFFLGNBQWM7cUJBQ3hCO2lCQUNGLENBQUMsQ0FBQztnQkFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakQsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGFBQWEsRUFBRSxVQUFVLEVBQUU7b0JBQ3BFLE1BQU0sRUFBRTt3QkFDTjs0QkFDRSxTQUFTLEVBQUUsUUFBUTs0QkFDbkIsT0FBTyxFQUFFLENBQUMsSUFBSSxTQUFTLENBQUMsc0JBQXNCLENBQUM7b0NBQzdDLFVBQVUsRUFBRSxZQUFZO29DQUN4QixVQUFVLEVBQUUsVUFBVSxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsTUFBTSxFQUFFLFdBQVcsQ0FBQztvQ0FDeEYsTUFBTSxFQUFFLFlBQVk7aUNBQ3JCLENBQUMsQ0FBQzt5QkFDSjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQztvQkFDbkMsU0FBUyxFQUFFLE9BQU87aUJBQ25CLENBQUMsQ0FBQztnQkFFSCxnQ0FBZ0M7Z0JBQ2hDLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUNqRCxVQUFVLEVBQUUsUUFBUTtvQkFDcEIsS0FBSyxFQUFFLFlBQVk7b0JBQ25CLE9BQU87aUJBQ1IsQ0FBQyxDQUFDLENBQUM7Z0JBRUosTUFBTSxZQUFZLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUNqRCxVQUFVLEVBQUUsUUFBUTtvQkFDcEIsS0FBSyxFQUFFLFlBQVk7b0JBQ25CLE9BQU87b0JBQ1AsT0FBTyxFQUFFLENBQUMsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7aUJBQ3ZDLENBQUMsQ0FBQztnQkFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDZixVQUFVLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUNyQyxDQUFDLEVBQUUsbURBQW1ELENBQUMsQ0FBQztnQkFFeEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBRUQsc0NBQXNDLENBQUMsSUFBVTtZQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sZ0JBQWdCLEdBQUcsU0FBUyxDQUFDLGVBQWUsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFDbkYsc0NBQXNDLENBQUMsQ0FBQztZQUUxQyxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNqRCxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtnQkFDM0MsTUFBTSxFQUFFO29CQUNOO3dCQUNFLFNBQVMsRUFBRSxRQUFRO3dCQUNuQixPQUFPLEVBQUU7NEJBQ1AsSUFBSSxTQUFTLENBQUMsY0FBYyxDQUFDO2dDQUMzQixVQUFVLEVBQUUsV0FBVztnQ0FDdkIsTUFBTSxFQUFFLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDO2dDQUN0QyxTQUFTLEVBQUUsS0FBSztnQ0FDaEIsTUFBTSxFQUFFLFlBQVk7NkJBQ3JCLENBQUM7eUJBQ0g7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsU0FBUyxFQUFFLE9BQU87d0JBQ2xCLE9BQU8sRUFBRTs0QkFDUCxJQUFJLFNBQVMsQ0FBQyxlQUFlLENBQUM7Z0NBQzVCLFVBQVUsRUFBRSxXQUFXO2dDQUN2QixLQUFLLEVBQUUsWUFBWTtnQ0FDbkIsT0FBTyxFQUFFLGdCQUFnQjs2QkFDMUIsQ0FBQzt5QkFDSDtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7Z0JBQy9ELFFBQVEsRUFBRTtvQkFDUjt3QkFDRSxNQUFNLEVBQUUsUUFBUTtxQkFDakI7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLE1BQU0sRUFBRSxXQUFXO2dDQUNuQixlQUFlLEVBQUU7b0NBQ2YsYUFBYSxFQUFFLHNDQUFzQztpQ0FDdEQ7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxnREFBZ0QsQ0FBQyxJQUFVO1lBQ3pELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFMUIsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDakQsTUFBTSxlQUFlLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDO2dCQUNwRCxVQUFVLEVBQUUsV0FBVztnQkFDdkIsS0FBSyxFQUFFLFlBQVk7Z0JBQ25CLE9BQU8sRUFBRSxJQUFJLFNBQVMsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtvQkFDekQsU0FBUyxFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO3dCQUN4QyxPQUFPLEVBQUUsS0FBSzt3QkFDZCxHQUFHLEVBQUU7NEJBQ0gsb0JBQW9CLEVBQUU7Z0NBQ3BCLFNBQVM7NkJBQ1Y7eUJBQ0Y7d0JBQ0QsTUFBTSxFQUFFOzRCQUNOLEtBQUssRUFBRTtnQ0FDTCxRQUFRLEVBQUU7b0NBQ1IsNkJBQTZCO2lDQUM5Qjs2QkFDRjt5QkFDRjtxQkFDRixDQUFDO2lCQUNILENBQUM7YUFDSCxDQUFDLENBQUM7WUFDSCxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtnQkFDM0MsTUFBTSxFQUFFO29CQUNOO3dCQUNFLFNBQVMsRUFBRSxRQUFRO3dCQUNuQixPQUFPLEVBQUU7NEJBQ1AsSUFBSSxTQUFTLENBQUMsY0FBYyxDQUFDO2dDQUMzQixVQUFVLEVBQUUsV0FBVztnQ0FDdkIsTUFBTSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsUUFBUSxDQUFDO2dDQUMzRCxTQUFTLEVBQUUsS0FBSztnQ0FDaEIsTUFBTSxFQUFFLFlBQVk7NkJBQ3JCLENBQUM7eUJBQ0g7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsU0FBUyxFQUFFLE9BQU87d0JBQ2xCLE9BQU8sRUFBRTs0QkFDUCxlQUFlOzRCQUNmLElBQUksU0FBUyxDQUFDLG9CQUFvQixDQUFDO2dDQUNqQyxVQUFVLEVBQUUsU0FBUztnQ0FDckIscUJBQXFCLEVBQUUsZUFBZSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUM7Z0NBQzFELGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsNENBQTRDLENBQUM7Z0NBQ3ZHLFFBQVEsRUFBRSxDQUFDOzZCQUNaLENBQUM7eUJBQ0g7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO2dCQUMvRCxRQUFRLEVBQUU7b0JBQ1I7d0JBQ0UsTUFBTSxFQUFFLFFBQVE7cUJBQ2pCO29CQUNEO3dCQUNFLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFNBQVMsRUFBRTs0QkFDVDtnQ0FDRSxNQUFNLEVBQUUsV0FBVztnQ0FDbkIsV0FBVyxFQUFFLG9CQUFvQjs2QkFDbEM7NEJBQ0Q7Z0NBQ0UsTUFBTSxFQUFFLFNBQVM7Z0NBQ2pCLGVBQWUsRUFBRTtvQ0FDZixZQUFZLEVBQUUsK0JBQStCO2lDQUM5Qzs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gXCJAYXdzLWNkay9hc3NlcnRcIjtcbmltcG9ydCAqIGFzIGNvZGVidWlsZCBmcm9tICdAYXdzLWNkay9hd3MtY29kZWJ1aWxkJztcbmltcG9ydCAqIGFzIGNvZGVjb21taXQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVjb21taXQnO1xuaW1wb3J0ICogYXMgY29kZXBpcGVsaW5lIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdAYXdzLWNkay9hd3Mtc25zJztcbmltcG9ydCB7IEFwcCwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBjcGFjdGlvbnMgZnJvbSAnLi4vLi4vbGliJztcblxuLy8gdHNsaW50OmRpc2FibGU6b2JqZWN0LWxpdGVyYWwta2V5LXF1b3Rlc1xuXG5leHBvcnQgPSB7XG4gICdDb2RlQnVpbGQgYWN0aW9uJzoge1xuICAgICd0aGF0IGlzIGNyb3NzLWFjY291bnQgYW5kIGhhcyBvdXRwdXRzJzoge1xuICAgICAgJ2NhdXNlcyBhbiBlcnJvcicodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICAgICAgY29uc3QgcHJvamVjdFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1Byb2plY3RTdGFjaycsIHtcbiAgICAgICAgICBlbnY6IHtcbiAgICAgICAgICAgIHJlZ2lvbjogJ3VzLXdlc3QtMicsXG4gICAgICAgICAgICBhY2NvdW50OiAnMDEyMzQ1Njc4OTEyJyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgcHJvamVjdCA9IG5ldyBjb2RlYnVpbGQuUGlwZWxpbmVQcm9qZWN0KHByb2plY3RTdGFjaywgJ1Byb2plY3QnKTtcblxuICAgICAgICBjb25zdCBwaXBlbGluZVN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1BpcGVsaW5lU3RhY2snLCB7XG4gICAgICAgICAgZW52OiB7XG4gICAgICAgICAgICByZWdpb246ICd1cy13ZXN0LTInLFxuICAgICAgICAgICAgYWNjb3VudDogJzAxMjM0NTY3ODkxMycsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHBpcGVsaW5lU3RhY2ssICdQaXBlbGluZScsIHtcbiAgICAgICAgICBzdGFnZXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc3RhZ2VOYW1lOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBjcGFjdGlvbnMuQ29kZUNvbW1pdFNvdXJjZUFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0NvZGVDb21taXQnLFxuICAgICAgICAgICAgICAgIHJlcG9zaXRvcnk6IGNvZGVjb21taXQuUmVwb3NpdG9yeS5mcm9tUmVwb3NpdG9yeU5hbWUocGlwZWxpbmVTdGFjaywgJ1JlcG8nLCAncmVwby1uYW1lJyksXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGJ1aWxkU3RhZ2UgPSBwaXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgICAgc3RhZ2VOYW1lOiAnQnVpbGQnLFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyB0aGlzIHdvcmtzIGZpbmUgLSBubyBvdXRwdXRzIVxuICAgICAgICBidWlsZFN0YWdlLmFkZEFjdGlvbihuZXcgY3BhY3Rpb25zLkNvZGVCdWlsZEFjdGlvbih7XG4gICAgICAgICAgYWN0aW9uTmFtZTogJ0J1aWxkMScsXG4gICAgICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICBwcm9qZWN0LFxuICAgICAgICB9KSk7XG5cbiAgICAgICAgY29uc3QgYnVpbGRBY3Rpb24yID0gbmV3IGNwYWN0aW9ucy5Db2RlQnVpbGRBY3Rpb24oe1xuICAgICAgICAgIGFjdGlvbk5hbWU6ICdCdWlsZDInLFxuICAgICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgcHJvamVjdCxcbiAgICAgICAgICBvdXRwdXRzOiBbbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgpXSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIGJ1aWxkU3RhZ2UuYWRkQWN0aW9uKGJ1aWxkQWN0aW9uMik7XG4gICAgICAgIH0sIC9odHRwczpcXC9cXC9naXRodWJcXC5jb21cXC9hd3NcXC9hd3MtY2RrXFwvaXNzdWVzXFwvNDE2OS8pO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuXG4gICAgJ2NhbiBiZSBiYWNrZWQgYnkgYW4gaW1wb3J0ZWQgcHJvamVjdCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgY29uc3QgY29kZUJ1aWxkUHJvamVjdCA9IGNvZGVidWlsZC5QaXBlbGluZVByb2plY3QuZnJvbVByb2plY3ROYW1lKHN0YWNrLCAnQ29kZUJ1aWxkJyxcbiAgICAgICAgJ2NvZGVCdWlsZFByb2plY3ROYW1lSW5Bbm90aGVyQWNjb3VudCcpO1xuXG4gICAgICBjb25zdCBzb3VyY2VPdXRwdXQgPSBuZXcgY29kZXBpcGVsaW5lLkFydGlmYWN0KCk7XG4gICAgICBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgIHN0YWdlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgIG5ldyBjcGFjdGlvbnMuUzNTb3VyY2VBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdTM19Tb3VyY2UnLFxuICAgICAgICAgICAgICAgIGJ1Y2tldDogbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0J1Y2tldCcpLFxuICAgICAgICAgICAgICAgIGJ1Y2tldEtleTogJ2tleScsXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgbmV3IGNwYWN0aW9ucy5Db2RlQnVpbGRBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdDb2RlQnVpbGQnLFxuICAgICAgICAgICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgICAgcHJvamVjdDogY29kZUJ1aWxkUHJvamVjdCxcbiAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgIFwiU3RhZ2VzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIk5hbWVcIjogXCJTb3VyY2VcIixcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiTmFtZVwiOiBcIkJ1aWxkXCIsXG4gICAgICAgICAgICBcIkFjdGlvbnNcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJOYW1lXCI6IFwiQ29kZUJ1aWxkXCIsXG4gICAgICAgICAgICAgICAgXCJDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgICAgICAgICAgICAgIFwiUHJvamVjdE5hbWVcIjogXCJjb2RlQnVpbGRQcm9qZWN0TmFtZUluQW5vdGhlckFjY291bnRcIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2V4cG9zZXMgdmFyaWFibGVzIGZvciBvdGhlciBhY3Rpb25zIHRvIGNvbnN1bWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgIGNvbnN0IGNvZGVCdWlsZEFjdGlvbiA9IG5ldyBjcGFjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uKHtcbiAgICAgICAgYWN0aW9uTmFtZTogJ0NvZGVCdWlsZCcsXG4gICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgIHByb2plY3Q6IG5ldyBjb2RlYnVpbGQuUGlwZWxpbmVQcm9qZWN0KHN0YWNrLCAnQ29kZUJ1aWxkJywge1xuICAgICAgICAgIGJ1aWxkU3BlYzogY29kZWJ1aWxkLkJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgICAgIHZlcnNpb246ICcwLjInLFxuICAgICAgICAgICAgZW52OiB7XG4gICAgICAgICAgICAgICdleHBvcnRlZC12YXJpYWJsZXMnOiBbXG4gICAgICAgICAgICAgICAgJ1NvbWVWYXInLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHBoYXNlczoge1xuICAgICAgICAgICAgICBidWlsZDoge1xuICAgICAgICAgICAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICAgICAgICAgICAnZXhwb3J0IFNvbWVWYXI9XCJTb21lIFZhbHVlXCInLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICB9KSxcbiAgICAgIH0pO1xuICAgICAgbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ1BpcGVsaW5lJywge1xuICAgICAgICBzdGFnZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdGFnZU5hbWU6ICdTb3VyY2UnLFxuICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICBuZXcgY3BhY3Rpb25zLlMzU291cmNlQWN0aW9uKHtcbiAgICAgICAgICAgICAgICBhY3Rpb25OYW1lOiAnUzNfU291cmNlJyxcbiAgICAgICAgICAgICAgICBidWNrZXQ6IHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ0J1Y2tldCcsICdidWNrZXQnKSxcbiAgICAgICAgICAgICAgICBidWNrZXRLZXk6ICdrZXknLFxuICAgICAgICAgICAgICAgIG91dHB1dDogc291cmNlT3V0cHV0LFxuICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdGFnZU5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgIGNvZGVCdWlsZEFjdGlvbixcbiAgICAgICAgICAgICAgbmV3IGNwYWN0aW9ucy5NYW51YWxBcHByb3ZhbEFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0FwcHJvdmUnLFxuICAgICAgICAgICAgICAgIGFkZGl0aW9uYWxJbmZvcm1hdGlvbjogY29kZUJ1aWxkQWN0aW9uLnZhcmlhYmxlKCdTb21lVmFyJyksXG4gICAgICAgICAgICAgICAgbm90aWZpY2F0aW9uVG9waWM6IHNucy5Ub3BpYy5mcm9tVG9waWNBcm4oc3RhY2ssICdUb3BpYycsICdhcm46YXdzOnNuczp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOm15dG9waWMnKSxcbiAgICAgICAgICAgICAgICBydW5PcmRlcjogMixcbiAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgIFwiU3RhZ2VzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIk5hbWVcIjogXCJTb3VyY2VcIixcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiTmFtZVwiOiBcIkJ1aWxkXCIsXG4gICAgICAgICAgICBcIkFjdGlvbnNcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJOYW1lXCI6IFwiQ29kZUJ1aWxkXCIsXG4gICAgICAgICAgICAgICAgXCJOYW1lc3BhY2VcIjogXCJCdWlsZF9Db2RlQnVpbGRfTlNcIixcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiTmFtZVwiOiBcIkFwcHJvdmVcIixcbiAgICAgICAgICAgICAgICBcIkNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgICAgICAgICAgXCJDdXN0b21EYXRhXCI6IFwiI3tCdWlsZF9Db2RlQnVpbGRfTlMuU29tZVZhcn1cIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxufTtcbiJdfQ==