"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
/**
 * A test stack with a half-prepared pipeline ready to add CloudFormation actions to
 */
class TestFixture extends cdk.Stack {
    constructor() {
        super();
        this.pipeline = new codepipeline.Pipeline(this, 'Pipeline');
        this.sourceStage = this.pipeline.addStage({ stageName: 'Source' });
        this.deployStage = this.pipeline.addStage({ stageName: 'Deploy' });
        this.repo = new codecommit.Repository(this, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        this.sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'Source',
            output: this.sourceOutput,
            repository: this.repo,
        });
        this.sourceStage.addAction(source);
    }
}
module.exports = {
    'CreateChangeSetAction can be used to make a change set from a CodePipeline'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'MagicPipeline');
        const changeSetExecRole = new aws_iam_1.Role(stack, 'ChangeSetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cloudformation.amazonaws.com'),
        });
        /** Source! */
        const repo = new codecommit.Repository(stack, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository: repo,
            trigger: cpactions.CodeCommitTrigger.POLL,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source]
        });
        /** Build! */
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        const buildOutput = new codepipeline.Artifact('OutputYo');
        const buildAction = new cpactions.CodeBuildAction({
            actionName: 'build',
            project,
            input: sourceOutput,
            outputs: [buildOutput],
        });
        pipeline.addStage({
            stageName: 'build',
            actions: [buildAction],
        });
        /** Deploy! */
        // To execute a change set - yes, you probably do need *:* 🤷‍♀️
        changeSetExecRole.addToPolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        const stackName = 'BrelandsStack';
        const changeSetName = 'MyMagicalChangeSet';
        pipeline.addStage({
            stageName: 'prod',
            actions: [
                new cpactions.CloudFormationCreateReplaceChangeSetAction({
                    actionName: 'BuildChangeSetProd',
                    stackName,
                    changeSetName,
                    deploymentRole: changeSetExecRole,
                    templatePath: new codepipeline.ArtifactPath(buildOutput, 'template.yaml'),
                    templateConfiguration: new codepipeline.ArtifactPath(buildOutput, 'templateConfig.json'),
                    adminPermissions: false,
                }),
                new cpactions.CloudFormationExecuteChangeSetAction({
                    actionName: 'ExecuteChangeSetProd',
                    stackName,
                    changeSetName,
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "ArtifactStore": {
                "Location": {
                    "Ref": "MagicPipelineArtifactsBucket212FE7BF"
                },
                "Type": "S3"
            }, "RoleArn": {
                "Fn::GetAtt": ["MagicPipelineRoleFB2BD6DE",
                    "Arn"
                ]
            },
            "Stages": [{
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Source",
                                "Owner": "AWS", "Provider": "CodeCommit", "Version": "1"
                            },
                            "Configuration": {
                                "RepositoryName": {
                                    "Fn::GetAtt": [
                                        "MyVeryImportantRepo11BC3EBD",
                                        "Name"
                                    ]
                                },
                                "BranchName": "master",
                                "PollForSourceChanges": true
                            },
                            "Name": "source",
                            "OutputArtifacts": [
                                {
                                    "Name": "SourceArtifact"
                                }
                            ],
                            "RunOrder": 1
                        }
                    ],
                    "Name": "source"
                },
                {
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Build",
                                "Owner": "AWS",
                                "Provider": "CodeBuild",
                                "Version": "1"
                            },
                            "Configuration": {
                                "ProjectName": {
                                    "Ref": "MyBuildProject30DB9D6E"
                                }
                            },
                            "InputArtifacts": [
                                {
                                    "Name": "SourceArtifact"
                                }
                            ],
                            "Name": "build",
                            "OutputArtifacts": [
                                {
                                    "Name": "OutputYo"
                                }
                            ],
                            "RunOrder": 1
                        }
                    ],
                    "Name": "build"
                },
                {
                    "Actions": [
                        {
                            "ActionTypeId": {
                                "Category": "Deploy",
                                "Owner": "AWS",
                                "Provider": "CloudFormation",
                                "Version": "1"
                            },
                            "Configuration": {
                                "ActionMode": "CHANGE_SET_REPLACE",
                                "ChangeSetName": "MyMagicalChangeSet",
                                "RoleArn": {
                                    "Fn::GetAtt": [
                                        "ChangeSetRole0BCF99E6",
                                        "Arn"
                                    ]
                                },
                                "StackName": "BrelandsStack",
                                "TemplatePath": "OutputYo::template.yaml",
                                "TemplateConfiguration": "OutputYo::templateConfig.json"
                            },
                            "InputArtifacts": [{ "Name": "OutputYo" }],
                            "Name": "BuildChangeSetProd",
                            "RunOrder": 1
                        },
                        {
                            "ActionTypeId": {
                                "Category": "Deploy",
                                "Owner": "AWS",
                                "Provider": "CloudFormation",
                                "Version": "1"
                            },
                            "Configuration": {
                                "ActionMode": "CHANGE_SET_EXECUTE",
                                "ChangeSetName": "MyMagicalChangeSet"
                            },
                            "Name": "ExecuteChangeSetProd",
                            "RunOrder": 1
                        }
                    ],
                    "Name": "prod"
                }
            ]
        }));
        test.done();
    },
    'fullPermissions leads to admin role and full IAM capabilities with pipeline bucket+key read permissions'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: true,
        }));
        const roleId = "PipelineDeployCreateUpdateRole515CB7D4";
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "Capabilities": "CAPABILITY_NAMED_IAM",
                                "RoleArn": { "Fn::GetAtt": [roleId, "Arn"] },
                                "ActionMode": "CREATE_UPDATE",
                                "StackName": "MyStack",
                                "TemplatePath": "SourceArtifact::template.yaml"
                            },
                            "InputArtifacts": [{ "Name": "SourceArtifact" }],
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        // THEN: Role is created with full permissions
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        "Action": [
                            "s3:GetObject*",
                            "s3:GetBucket*",
                            "s3:List*",
                        ],
                        "Effect": "Allow",
                    },
                    {
                        "Action": [
                            "kms:Decrypt",
                            "kms:DescribeKey",
                        ],
                        "Effect": "Allow",
                    },
                    {
                        Action: "*",
                        Effect: 'Allow',
                        Resource: "*"
                    }
                ],
            },
            Roles: [{ Ref: roleId }]
        }));
        test.done();
    },
    'outputFileName leads to creation of output artifact'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            outputFileName: 'CreateResponse.json',
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "OutputArtifacts": [{ "Name": "CreateUpdate_MyStack_Artifact" }],
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'replaceOnFailure switches action type'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            replaceOnFailure: true,
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "ActionMode": "REPLACE_ON_FAILURE",
                            },
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'parameterOverrides are serialized as a string'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            parameterOverrides: {
                RepoName: stack.repo.repositoryName
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "ParameterOverrides": { "Fn::Join": ["", [
                                            "{\"RepoName\":\"",
                                            { "Fn::GetAtt": ["MyVeryImportantRepo11BC3EBD", "Name"] },
                                            "\"}"
                                        ]] }
                            },
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Action service role is passed to template'(test) {
        const stack = new TestFixture();
        const importedRole = aws_iam_1.Role.fromRoleArn(stack, 'ImportedRole', 'arn:aws:iam::000000000000:role/action-role');
        const freshRole = new aws_iam_1.Role(stack, 'FreshRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('magicservice')
        });
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'ImportedRoleAction',
            role: importedRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'FreshRoleAction',
            role: freshRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                {
                    "Name": "Source" /* don't care about the rest */
                },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Name": "ImportedRoleAction",
                            "RoleArn": "arn:aws:iam::000000000000:role/action-role"
                        },
                        {
                            "Name": "FreshRoleAction",
                            "RoleArn": {
                                "Fn::GetAtt": [
                                    "FreshRole472F6E18",
                                    "Arn"
                                ]
                            }
                        }
                    ]
                }
            ]
        }));
        test.done();
    },
    'Single capability is passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM
            ]
        }));
        const roleId = "PipelineDeployCreateUpdateRole515CB7D4";
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "Capabilities": "CAPABILITY_NAMED_IAM",
                                "RoleArn": { "Fn::GetAtt": [roleId, "Arn"] },
                                "ActionMode": "CREATE_UPDATE",
                                "StackName": "MyStack",
                                "TemplatePath": "SourceArtifact::template.yaml"
                            },
                            "InputArtifacts": [{ "Name": "SourceArtifact" }],
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Multiple capabilities are passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
                aws_cloudformation_1.CloudFormationCapabilities.AUTO_EXPAND
            ]
        }));
        const roleId = "PipelineDeployCreateUpdateRole515CB7D4";
        // THEN: Action in Pipeline has named IAM and AUTOEXPAND capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "Capabilities": "CAPABILITY_NAMED_IAM,CAPABILITY_AUTO_EXPAND",
                                "RoleArn": { "Fn::GetAtt": [roleId, "Arn"] },
                                "ActionMode": "CREATE_UPDATE",
                                "StackName": "MyStack",
                                "TemplatePath": "SourceArtifact::template.yaml"
                            },
                            "InputArtifacts": [{ "Name": "SourceArtifact" }],
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Empty capabilities is not passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NONE
            ]
        }));
        const roleId = "PipelineDeployCreateUpdateRole515CB7D4";
        // THEN: Action in Pipeline has no capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                { "Name": "Source" /* don't care about the rest */ },
                {
                    "Name": "Deploy",
                    "Actions": [
                        {
                            "Configuration": {
                                "RoleArn": { "Fn::GetAtt": [roleId, "Arn"] },
                                "ActionMode": "CREATE_UPDATE",
                                "StackName": "MyStack",
                                "TemplatePath": "SourceArtifact::template.yaml"
                            },
                            "InputArtifacts": [{ "Name": "SourceArtifact" }],
                            "Name": "CreateUpdate",
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'cross-account CFN Pipeline': {
        'correctly creates the deployment Role in the other account'(test) {
            const app = new cdk.App();
            const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                env: {
                    account: '234567890123',
                    region: 'us-west-2',
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommit',
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'RepoName'),
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Deploy',
                        actions: [
                            new cpactions.CloudFormationCreateUpdateStackAction({
                                actionName: 'CFN',
                                stackName: 'MyStack',
                                adminPermissions: true,
                                templatePath: sourceOutput.atPath('template.yaml'),
                                account: '123456789012',
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Deploy",
                        "Actions": [
                            {
                                "Name": "CFN",
                                "RoleArn": { "Fn::Join": ["", ["arn:", { "Ref": "AWS::Partition" },
                                            ":iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50",
                                        ]],
                                },
                                "Configuration": {
                                    "RoleArn": {
                                        "Fn::Join": ["", ["arn:", { "Ref": "AWS::Partition" },
                                                ":iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508",
                                            ]],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            // the pipeline's BucketPolicy should trust both CFN roles
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": ["", ["arn:", { "Ref": "AWS::Partition" },
                                            ":iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508",
                                        ]],
                                },
                            },
                        },
                        {
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": ["", ["arn:", { "Ref": "AWS::Partition" },
                                            ":iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50",
                                        ]],
                                },
                            },
                        },
                    ],
                },
            }));
            const otherStack = app.node.findChild('cross-account-support-stack-123456789012');
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                "RoleName": "pipelinestack-support-123loycfnactionrole56af64af3590f311bc50",
            }));
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                "RoleName": "pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508",
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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