"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age: ${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-max-age: ${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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