"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const custom_action_registration_1 = require("../custom-action-registration");
class BaseJenkinsProvider extends cdk.Construct {
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK,
     * or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider
     * @param id the identifier of the new provider Construct
     * @param attrs the properties used to identify the existing provider
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5
};
//# sourceMappingURL=data:application/json;base64,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