"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const iam = require("aws-cdk-lib/aws-iam");
// import * as defaults from '@aws-solutions-constructs/core';
require("@aws-cdk/assert/jest");
const iotTopicRuleProps = {
    topicRulePayload: {
        description: "sends data to kinesis streams",
        sql: "SELECT * FROM 'solutions/constructs'",
        actions: []
    }
};
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "sends data to kinesis streams",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        RetentionPeriodHours: 24,
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check existing kinesis stream', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {
        shardCount: 2,
        streamName: 'testexistingstream',
        retentionPeriod: cdk.Duration.hours(25)
    });
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 25,
        Name: 'testexistingstream'
    });
    expect(stack).not.toHaveResource('AWS::CloudWatch::Alarm');
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check new kinesis stream with override props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps,
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    const kinesisStream = construct.kinesisStream;
    expect(kinesisStream).toBeDefined();
});
test('check existing action in topic rule props', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const existingIamRole = new iam.Role(stack, 'existingRole', {
        assumedBy: new iam.ServicePrincipal('iot.amazonaws.com')
    });
    const existingKinesisActionProperty = {
        kinesis: {
            streamName: existingKinesisStream.streamName,
            roleArn: existingIamRole.roleArn
        }
    };
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: [existingKinesisActionProperty],
                ruleDisabled: true
            },
            ruleName: 'testiotrulename'
        },
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    // Check multiple actions in the Topic Rule
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                },
                {
                    Kinesis: {
                        RoleArn: {
                            "Fn::GetAtt": [
                                "existingRole3E995BBA",
                                "Arn"
                            ]
                        },
                        StreamName: {
                            Ref: "existingstream0A902451"
                        }
                    }
                }
            ],
            RuleDisabled: true,
            Sql: "SELECT * FROM 'solutions/constructs'"
        },
        RuleName: "testiotrulename"
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check name confict', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props);
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check construct chaining', () => {
    const stack = new cdk.Stack();
    const props1 = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props1);
    const props2 = {
        iotTopicRuleProps,
        existingStreamObj: construct.kinesisStream
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props2);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 1);
});
test('check error when stream props and existing stream is supplied', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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