import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * @stability stable
 */
export interface IRepository extends IResource {
    /**
     * The ARN of this Repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryArn: string;
    /**
     * The human-visible name of this Repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryName: string;
    /**
     * The HTTP clone URL.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryCloneUrlHttp: string;
    /**
     * The SSH clone URL.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryCloneUrlSsh: string;
    /**
     * The HTTPS (GRC) clone URL.
     *
     * HTTPS (GRC) is the protocol to use with git-remote-codecommit (GRC).
     *
     * It is the recommended method for supporting connections made with federated
     * access, identity providers, and temporary credentials.
     *
     * @see https://docs.aws.amazon.com/codecommit/latest/userguide/setting-up-git-remote-codecommit.html
     * @stability stable
     */
    readonly repositoryCloneUrlGrc: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     *
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     *
     * @stability stable
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     *
     * @stability stable
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     *
     * @stability stable
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     *
     * @stability stable
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     *
     * @stability stable
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     *
     * @stability stable
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read this repository.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Options for the onCommit() method.
 *
 * @stability stable
 */
export interface OnCommitOptions extends events.OnEventOptions {
    /**
     * The branch to monitor.
     *
     * @default - All branches
     * @stability stable
     */
    readonly branches?: string[];
}
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
declare abstract class RepositoryBase extends Resource implements IRepository {
    /** The ARN of this Repository. */
    abstract readonly repositoryArn: string;
    /** The human-visible name of this Repository. */
    abstract readonly repositoryName: string;
    /** The HTTP clone URL */
    abstract readonly repositoryCloneUrlHttp: string;
    /** The SSH clone URL */
    abstract readonly repositoryCloneUrlSsh: string;
    abstract readonly repositoryCloneUrlGrc: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     *
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     *
     * @stability stable
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     *
     * @stability stable
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     *
     * @stability stable
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     *
     * @stability stable
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     *
     * @stability stable
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     *
     * @stability stable
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read this repository.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
/**
 * @stability stable
 */
export interface RepositoryProps {
    /**
     * Name of the repository.
     *
     * This property is required for all CodeCommit repositories.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * A description of the repository.
     *
     * Use the description to identify the
     * purpose of the repository.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
}
/**
 * Provides a CodeCommit Repository.
 *
 * @stability stable
 */
export declare class Repository extends RepositoryBase {
    /**
     * Imports a codecommit repository.
     *
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`).
     * @stability stable
     */
    static fromRepositoryArn(scope: Construct, id: string, repositoryArn: string): IRepository;
    /**
     * @stability stable
     */
    static fromRepositoryName(scope: Construct, id: string, repositoryName: string): IRepository;
    /**
     * The ARN of this Repository.
     *
     * @stability stable
     */
    readonly repositoryArn: string;
    /**
     * The human-visible name of this Repository.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * The HTTP clone URL.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlHttp: string;
    /**
     * The SSH clone URL.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlSsh: string;
    /**
     * The HTTPS (GRC) clone URL.
     *
     * HTTPS (GRC) is the protocol to use with git-remote-codecommit (GRC).
     *
     * It is the recommended method for supporting connections made with federated
     * access, identity providers, and temporary credentials.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlGrc: string;
    private readonly triggers;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RepositoryProps);
    /**
     * Create a trigger to notify another service to run actions on repository events.
     *
     * @param arn Arn of the resource that repository events will notify.
     * @param options Trigger options to run actions.
     * @stability stable
     */
    notify(arn: string, options?: RepositoryTriggerOptions): Repository;
}
/**
 * Creates for a repository trigger to an SNS topic or Lambda function.
 *
 * @stability stable
 */
export interface RepositoryTriggerOptions {
    /**
     * A name for the trigger.Triggers on a repository must have unique names.
     *
     * @stability stable
     */
    readonly name?: string;
    /**
     * The repository events for which AWS CodeCommit sends information to the target, which you specified in the DestinationArn property.If you don't specify events, the trigger runs for all repository events.
     *
     * @stability stable
     */
    readonly events?: RepositoryEventTrigger[];
    /**
     * The names of the branches in the AWS CodeCommit repository that contain events that you want to include in the trigger.
     *
     * If you don't specify at
     * least one branch, the trigger applies to all branches.
     *
     * @stability stable
     */
    readonly branches?: string[];
    /**
     * When an event is triggered, additional information that AWS CodeCommit includes when it sends information to the target.
     *
     * @stability stable
     */
    readonly customData?: string;
}
/**
 * Repository events that will cause the trigger to run actions in another service.
 *
 * @stability stable
 */
export declare enum RepositoryEventTrigger {
    /**
     * @stability stable
     */
    ALL = "all",
    /**
     * @stability stable
     */
    UPDATE_REF = "updateReference",
    /**
     * @stability stable
     */
    CREATE_REF = "createReference",
    /**
     * @stability stable
     */
    DELETE_REF = "deleteReference"
}
export {};
