# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['misty2py', 'misty2py.basic_skills', 'misty2py.utils']

package_data = \
{'': ['*']}

install_requires = \
['python-dotenv>=0.17.1,<0.18.0',
 'requests>=2.25.1,<3.0.0',
 'websocket-client>=0.58.0,<0.59.0']

setup_kwargs = {
    'name': 'misty2py',
    'version': '4.2.0',
    'description': 'A Python library for Misty II development',
    'long_description': '# Misty2py\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black) [![GitHub license](https://img.shields.io/github/license/Naereen/StrapDown.js.svg)](https://github.com/ChrisScarred/misty2py/blob/main/LICENSE)\n\nMisty2py is a Python 3 package for Misty II development using [Misty\'s REST API](https://docs.mistyrobotics.com/misty-ii/rest-api/api-reference/ "Misty Robotics REST API").\n\n## Installation\n\n### Poetry\n\nTo install misty2py, run `pip install misty2py`.\n\n### From source\n\n- If this is your first time using `misty2py` from source, do following:\n\n  - Get Poetry (`python -m pip install poetry`) if you do not have it yet.\n  - Copy `.env.example` to `.env`.\n  - Replace the placeholder values in the new `.env` file.\n  - Run `poetry install` to obtain all dependencies.\n\n- Run the desired script via `poetry run python -m [name]` where `[name]` is the placeholder for the module location (in Python notation).\n- If the scripts run but your Misty does not seem to respond, you have most likely provided an incorrect IP address for `MISTY_IP_ADDRESS` in `.env`.\n- Pytests can be run via `poetry run pytest .`.\n- The coverage report can be obtained via `poetry run pytest --cov-report html --cov=misty2py tests` for HTML output or via `poetry run pytest --cov=misty2py tests` for terminal output.\n\n## Features\n\nMisty2py can be used to develop complex skills (behaviours) for the Misty II robot utilising:\n\n- **actions** via sending a `POST` or `DELETE` requests to Misty\'s API;\n- **informations** via sending a `GET` request to Misty\'s API;\n- **continuous streams of data** via subscribing to event types on Misty\'s websockets.\n\nMisty2py uses following concepts for easy of usage:\n\n- **action keywords** - customisable python-styled keywords for endpoints of Misty\'s API that correspond to performing actions;\n- **information keywords** - customisable python-styled keywords for endpoints of Misty\'s API that correspond to retrieving information;\n- **data shortcuts** - customisable python-styled keywords for commonly used data that are supplied to Misty\'s API as the body of a `POST` request.\n\n## Usage\n\n### Getting started\n\n- Start by making **a new instance** of `misty2py.robot`\'s `Misty` by `misty_robot = Misty("ip_address_here")`\n  - Substitute `ip_address_here` with the IP address of your Misty.\n  - `misty2py.utils.env_loader` module contains `EnvLoader` class that can be used to load Misty\'s IP from the .env file in your project\'s home directory:\n    - Create the `.env` file and write `MISTY_IP_ADDRESS="[ip_address_here]"` in it.\n    - Initialise an `EnvLoader` object via `env_loader = EnvLoader()`; this loads the environment variables.\n    - Use `env_loader.get_ip()` to obtain the IP address.\n- Use the method `misty_robot.perform_action()` to tell Misty to **perform an action**.\n- Use the method `misty_robot.get_info()` to tell Misty to **return information**.\n- Use the method `misty_robot.event()` to initialise, obtain and stop **continuous streams of data** from Misty\'s event types.\n\n### Obtaining information\n\nObtaining digital information is handled by `misty2py.robot::get_info` method.\n\n`misty2py.robot::get_info` has following arguments:\n\n- `info_name` - *required;* the string information keyword corresponding to an endpoint in Misty\'s API;\n- `params` - *optional;* a dictionary of parameter name and parameter value pairs, defaults to `{}`.\n\n### Performing actions\n\nPerforming physical and digital actions including removal of non-system files is handled by `misty2py.robot::perform_action()` method.\n\n`misty2py.robot::perform_action()` has following arguments:\n\n- `action_name` - *required;* the string action keyword corresponding to an endpoint in Misty\'s API;\n- `data` - *optional;* the data to pass to the request as a dictionary or a data shortcut (string), defaults to `{}`.\n\n### Event types\n\nTo obtain event data in Misty\'s framework, it is required to **subscribe** to an event type on Misty\'s websocket server. Misty\'s websocket server then streams data to the websocket client, in this implementation via a separate thread. To **access this data,** `misty2py.robot::event` method must be called with `"get_data"` parameter from the main thread. When data are no longer required to be streamed to the client, an event type can be **unsubscribed** to kill the event thread.\n\n#### Subscription\n\nSubscribe to an event via `misty2py.robot::event` with the parameter `"subscribe"` and following keyword arguments:\n\n    - `type` - *required;* event type string as documented in [Event Types Docs](https://docs.mistyrobotics.com/misty-ii/robot/sensor-data/ "Misty Robotics Event Types").\n    - `name` - *optional;* a custom event name string; must be unique.\n    - `return_property` - *optional;* the property to return from Misty\'s websockets; all properties are returned if return_property is not supplied.\n    - `debounce` - *optional;* the interval in ms at which new information is sent; defaults to 250.\n    - `len_data_entries` - *optional;* the maximum number of data entries to keep (discards in fifo style); defaults to 10.\n    - `event_emitter` - *optional;* an event emitter function which emits an event upon message recieval. Supplies the message content as an argument.\n\n#### Accessing the data and the log\n\nAccess the data of an event or its log via `misty2py.robot::event` with the parameter `"get_data"` or `"get_log"` and a keyword argument `name` (the name of the event).\n\n#### Unsubscribing\n\nUnsubscribe from an event via `misty2py.robot::event` with the parameter `"unsubscribe"` and a keyword argument `name` (the name of the event).\n\n#### Basic example\n\n```python\nimport time\n\nfrom misty2py.robot import Misty\nfrom misty2py.utils.env_loader import EnvLoader\n\nenv_loader = EnvLoader\n\nm = Misty(env_loader.get_ip())\n\nd = m.event("subscribe", type = "BatteryCharge")\ne_name = d.get("event_name")\n\ntime.sleep(1)\n\nd = m.event("get_data", name = e_name)\n\nd = m.event("unsubscribe", name = e_name)\n```\n\n#### Event emitter usage - example\n\n```python\nimport time\nfrom pymitter import EventEmitter\n\nfrom misty2py.robot import Misty\nfrom misty2py.utils.env_loader import EnvLoader\n\nenv_loader = EnvLoader\n\nm = Misty(env_loader.get_ip())\nee = EventEmitter()\nevent_name = "myevent_001"\n\n@ee.on(event_name)\ndef listener(data):\n    print(data)\n\nd = m.event("subscribe", type = "BatteryCharge", name = event_name, event_emitter = ee)\n\ntime.sleep(2)\n\nd = m.event("unsubscribe", name = event_name)\n```\n\n### Adding custom keywords and shortcuts\n\nCustom keywords and shortcuts can be passed to a Misty object while declaring a new instance by using the optional arguments:\n\n- `custom_info` for custom information keywords (a dictionary with keys being the information keywords and values being the endpoints),\n- `custom_actions` for custom action keywords (a dictionary with keys being the action keywords and values being a dictionary `{"endpoint" : "edpoint_value", "method" : "method_value"}` where `method_value` is either `post` or `delete`),\n- `custom_data` for custom data shortcuts (a dictionary with keys being the data shortcuts and values being the dictionary of data values).\n\nAn example:\n\n```python\ncustom_allowed_infos = {\n    "hazards_settings": "api/hazards/settings"\n}\n\ncustom_allowed_data = {\n    "amazement": {\n        "FileName": "s_Amazement.wav"\n    },\n    "red": {\n        "red": "255",\n        "green": "0",\n        "blue": "0"\n    }\n}\n\ncustom_allowed_actions = {\n    "audio_play" : {\n        "endpoint" : "api/audio/play",\n        "method" : "post"\n    },\n    "delete_audio" : {\n        "endpoint" : "api/audio",\n        "method" : "delete"\n    }\n}\n\nmisty_robot = Misty("0.0.0.0", \n    custom_info=custom_allowed_infos, \n    custom_actions=custom_allowed_actions, \n    custom_data=custom_allowed_data)\n```\n',
    'author': "Chris 'Scar(R)ed' Vajdík",
    'author_email': 'scarred.chris@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ChrisScarred/misty2py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
