#!/usr/bin/env python3
"""
Functions for generating summary tables and plots
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import os
try:
    import ERVsearch.HelperFunctions as HelperFunctions
    import ERVsearch.Fasta as Fasta
    import ERVsearch.ORFs as ORFs
except ImportError:
    import HelperFunctions
    import Fasta
    import ORFs


def getInfiles(allinfiles, gene, ind):
    '''
    Get the input files corresponding to a particular index and gene - just
    looks through the list and returns files starting with the gene name
    '''
    # allinfiles is sometimes a list and sometimes a list of lists
    if isinstance(allinfiles[0], list):
        inf = np.array([i[ind] for i in allinfiles])
    else:
        inf = np.array(allinfiles)
    # get just the basename
    inf_stems = np.array([os.path.basename(x) for x in inf])
    # does it start with the gene name
    infile = inf[np.char.startswith(inf_stems, gene)][0]
    return (infile)


def getOutfiles(D, plot_format):
    '''
    Returns the output file names for the summary plots as a list by
    iterating through the dictionary generated by allScreenOutfiles and
    adding the directory paths and suffixes.
    '''
    outfiles = []
    for section in D:
        o = []
        for suffix in D[section]:
            # tables go in the summary tables directory
            if suffix.endswith(".txt"):
                o.append("summary_tables.dir/%s_%s" % (section,
                                                       suffix)),
            else:
                # plots got in the summary plots directory
                # add the user specified plot format
                o.append("summary_plots.dir/%s_%s.%s" % (section,
                                                         suffix,
                                                         plot_format))
        outfiles.append(o)
    return (outfiles)


def allScreenOutfiles(plot_format):
    '''
    Dictionary of the output file names for the summary plots for the Screen
    functions. Just used to save typing and tidy up because there are often
    several with the same prefix and suffix.
    '''
    D = {'exonerate_initial': ['summary.txt',
                               'lengths', 'scores', 'strands',
                               'by_sequence', 'counts_per_gene'],
         'ublast_hits': ['initial_summary.txt',
                         'alignment_length', 'perc_similarity',
                         'bit_score', 'by_match', 'per_gene'],
         'orfs': ['initial_summary.txt', 'lengths', 'strands',
                  'by_gene'],
         'ublast_orfs': ['initial_summary.txt',
                         'alignment_length',
                         'perc_similarity',
                         'bit_score',
                         'by_match',
                         'per_gene']}
    outfiles = getOutfiles(D, plot_format)
    return (outfiles)


def makeInitialFigures(n, dpi, by=None):
    '''
    Makes a list of n figures with the specified dpi, by is a list of widths
    for the figures, if by is not specified they are all 4 inches wide.
    '''
    figs = []
    # if by is not specified it should be 4 for all figures
    if not by:
        by = [4] * n
    for i in np.arange(n):
        figs.append(plt.figure(figsize=(n*by[i], by[i]), dpi=int(dpi)))
    return (figs)


def saveFigures(figs, outfiles, dpi):
    '''
    Saves each figure to file at the right dpi and with bbox_inches='tight'
    '''
    for i, f in enumerate(figs):
        f.savefig(outfiles[i], dpi=int(dpi), bbox_inches='tight')
        # Close the figure or matplotlib runs out of memory
        plt.close()


def setMax(figure, m, x=True, y=True):
    '''
    Used to make the axis limits the same for multiple plots in a figure.
    m is a dictionary where x is a list of minimum and maximum x values from
    the subplots and y a list of minimum and maximum y values.
    If either is None that axis is ignored.
    '''
    # iterate through the subplots
    for ax in figure.get_axes():
        if x:
            ax.set_xlim(min(m['x']), max(m['x'])*1.1)
        if y:
            ax.set_ylim(0, max(m['y'])*1.1)


def roundAll(tab):
    '''
    Rounds columns in a table which are floats to two decimal places.
    '''
    for col in tab.columns:
        if tab[col].dtype == 'float':
            tab[col] = tab[col].round(2)
    return (tab)


def makeSummaryD(table_file,
                 columns, header, columns_to_plot, group_by, maketab=False,
                 keep=""):
    '''
    Makes a dictionary summarising the information in the file table_file.
    Table_file can either be the path to a dataframe (default) or a Fasta
    file (if maketab is True). If table_file is a table, columns is the
    column names to add (if header=None) and header is either 0 if the table
    has a header or None if it does not.

    columns_to_plot is a list of columns to summarise.

    group_by is a column to group the table by and count how many are in
    each group, e.g. if group_by is genus this will count how many
    rows in the table are of each genus, then calculate the mean, maximum
    and minimum number per genus.

    If maketab is True, the function expects a Fasta file where sequence
    names are formated as ID_chromosome_start_end_(strand) and this information
    is used to make a dataframe. In this case, only sequences with the keep
    string in the name are included.
    '''
    # If a table needs to be created (if the input is a FASTA file)
    if maketab:
        D = dict()
        # convert the fasta file into a dictionary
        F = Fasta.makeFastaDict(table_file)
        # iterate through it
        for nam in F.keys():
            # if the keep string is in the name
            if keep in nam:
                # split the name into a dictionary with chrom, start, end
                # strand and put this into D with the sequence name as the key
                D[nam] = ORFs.splitNam(nam)
        # Turn the dictionary into a dataframe
        tab = pd.DataFrame(D).T
    else:
        # Read the dataframe
        tab = pd.read_csv(table_file, sep="\t", header=header)
        # Add column names if needed
        if columns:
            tab.columns = columns
    # Sometimes the length column isn't there but is needed - make it by
    # subtracting start from end
    if 'length' in columns_to_plot:
        tab['start'] = tab['start'].astype(int)
        tab['end'] = tab['end'].astype(int)
        tab['length'] = tab['end'] - tab['start']
    # calculate the various stats for the summary table
    statD = dict()
    # total number of regions in the table
    statD['N_Regions_Identified'] = len(tab)
    for col in columns_to_plot:
        # summary statistics
        statD['Mean_%s' % col] = np.mean(tab[col])
        statD['Max_%s' % col] = np.max(tab[col])
        statD['Min_%s' % col] = np.max(tab[col])

    if group_by:
        # count the number of regions in each group
        grouped = tab.groupby(group_by).size()
        # Number of groups
        statD['Total_N_%s' % group_by] = len(set(tab[group_by]))

        # Mean, maximum and minimum number per group
        statD['Mean_by_%s' % group_by] = np.mean(grouped)
        statD['Max_by_%s' % group_by] = np.max(grouped)
        statD['Min_by_%s' % group_by] = np.min(grouped)

        # only return grouped if this parameter is specified.
        return (statD, tab, grouped)
    return (statD, tab)


def makePlots(plots, cols, colnams, titles, types, table, genes, plotparams,
              outfiles):
    '''
    Make the summary plots for the Screen functions. These are bar plots or
    histograms showing, for each gene, the information from a specified
    column in table.

    plots, cols, colnams, titles and types are all lists in the same order
    specifying parts of each plot

    * plots - a list of matplotlib figures
    * cols - the column to plot in each figure
    * colnams - column name formatted to use as an axis label
    * titles - titles to add to plots
    * types - plot type, can be hist (histogram) or bar (barplot)

    These are zipped together and iterated through.

    If typ is hist a histogram is generated showing the distribution of values
    in the specified column. If typ is bar a bar plot is generated showing
    the number of each possible value in the column

    Table is the dataframe which contains the columns to be plotted.
    '''
    # zip the various attributes of the plots into one object
    Z = zip(plots, cols, colnams, titles, types)

    # for each set of plot information
    for plot, col, colnam, title, typ in Z:
        # dictionary to store the axis limits
        maxes = {'x': [], 'y': []}

        # for each gene
        for i, gene in enumerate(genes):
            # make a new subplot
            sp = plot.add_subplot(1, len(genes), i+1)
            # histograms
            if typ == 'hist':
                # extract the right column for this gene
                h = table[col][table['gene'] == gene]
                # plot a histogram of the values in the column
                sp.hist(h, color=plotparams['%s_colour' % gene])
                # label the axes
                sp.set_xlabel(colnam)
                sp.set_ylabel("Frequency")

            # bar charts
            elif typ == 'bar':
                # sort so the order is consistent
                table = table.sort_values(col)

                # count how many rows have each possible value in this column
                h = table[table['gene'] == gene].groupby(col).size()

                # plot a bar chart of the counts
                sp.bar(np.arange(len(h)), h,
                       color=plotparams['%s_colour' % gene])

                # add a tick in the middle of each bar
                sp.set_xticks(np.arange(0, len(h)))

                # rotate the x axis labels if there are more than 3
                if len(h) > 3:
                    sp.set_xticklabels(h.index, rotation='vertical')
                else:
                    sp.set_xticklabels(h.index)

            # add the title
            sp.set_title(gene)

            # keep track of the minimum and maximum values on the axis
            maxes['x'].append(sp.get_xlim()[1])
            maxes['x'].append(sp.get_xlim()[0])
            maxes['y'].append(sp.get_ylim()[1])
            maxes['y'].append(sp.get_xlim()[0])
        plot.suptitle(title, y=1.05)

        # the user can specify to match the axis limits on the different gene
        # plots
        if plotparams['match_axes'] == "True":
            # for histograms set x and y as both are numeric
            if typ == 'hist':
                setMax(plot, maxes)
            elif typ == "bar":
                # for bar plots x is categorical
                setMax(plot, maxes, x=False)
        # arranges everything on the page
        plot.tight_layout()
    # force dpi to be an integer
    dpi = int(plotparams['dpi'])
    # save all the figures
    saveFigures(plots, outfiles, dpi=dpi)


def makeGenePlot(plot, bigtab, genes, title, plotparams, outfile):
    '''
    Generates a single bar chart with one column per gene according to the
    user specifications.
    '''
    # add the subplot
    sp = plot.add_subplot(111)
    for i, gene in enumerate(genes):
        # add the bars in the right colours
        sp.bar(i, len(bigtab[bigtab['gene'] == gene]),
               color=plotparams['%s_colour' % gene])

    # tidy up the axes
    sp.set_xticks(np.arange(i))
    sp.set_xticklabels(genes)
    sp.set_ylabel("Number of Regions")
    # add the title
    sp.set_title(title)
    # set the dpi
    dpi = int(plotparams['dpi'])
    # save
    saveFigures([plot], [outfile], dpi=dpi)


def summariseExonerateInitial(infiles, outfiles, log, genes, plotparams):
    '''
    Summarise the initial Exonerate run.
    Based on the merged regions in gene_bed_files.dir generated with
    mergeOverlaps.
    '''
    statD = dict()

    # make the empty figures
    dpi = int(plotparams['dpi'])
    lengths, counts, scores, tots, strands = makeInitialFigures(
        5, dpi=dpi)

    log.info("Plotting summary information for initial Exonerate run")
    bigtab = pd.DataFrame()
    biggrouped = pd.DataFrame()
    for i, gene in enumerate(genes):
        # get the files for the right gene
        bed = getInfiles(infiles, gene, 1)
        # retrieve summary stats for the length and bit_score columns
        # for this gene
        sD, tab, grouped = makeSummaryD(bed,
                                        HelperFunctions.getBedColumns(),
                                        header=None,
                                        columns_to_plot=['length',
                                                         'bit_score'],
                                        group_by='chrom')
        # store the summary stats
        statD[gene] = sD
        tab['gene'] = gene
        # store everything in a table
        bigtab = bigtab.append(tab)

        # store the table grouped by chromosome
        grouped = pd.DataFrame(grouped)
        grouped['gene'] = gene
        biggrouped = biggrouped.append(grouped)

    # fix the columns in the grouped table
    biggrouped.columns = ['count', 'gene']

    # plot everything in the ungrouped table
    makePlots([lengths, scores, strands],
              ['length', 'bit_score', 'strand'],
              ['Length (nt)', 'Exonerate Score', 'Strand'],
              ["Putative ERV Region Lengths - Exonerate Initial",
               "Putative ERV Region Scores - Exonerate Initial",
               "Putative ERV Region Strands - Exonerate Initial"],
              ['hist', 'hist', 'bar'],
              bigtab,
              genes,
              plotparams,
              [outfiles[1], outfiles[2], outfiles[3]])

    # plot everything in the grouped table
    makePlots([counts],
              ['count'],
              ['Regions per Sequence'],
              ['Number of ERV Regions Identified per Input Sequence - Exonerate Initial'],
              ['hist'],
              biggrouped,
              genes,
              plotparams,
              [outfiles[4]])

    # plot counts per gene
    makeGenePlot(tots, bigtab, genes, "Number of ERV Regions per Gene - Exonerate Initial",
                 plotparams, outfiles[5])

    # convert the summary stats into a dataframe and save
    df = pd.DataFrame(statD)
    df = roundAll(df)
    df.to_csv(outfiles[0], sep="\t")


def summariseUBLAST(infiles, outfiles, log, genes, plotparams):
    '''
    Summarise UBLAST output.
    Run on the UBLAST output from runUBLASTCheck and checkORFsUBLAST
    '''
    # make the empty figures
    dpi = int(plotparams['dpi'])
    # target needs to be wider because there's a column for each match
    tot, lengths, sim, score, target = makeInitialFigures(
        5, dpi=dpi, by=[4, 4, 4, 4, 8])
    log.info("Plotting summary information for UBLAST")
    statD = dict()
    bigtab = pd.DataFrame()
    for i, gene in enumerate(genes):
        # find the input files for this gene
        f = getInfiles(infiles, gene, 1)
        # calculate summary stats for alignment length, percentage identity
        # and bit score and per group for target
        sD, tab, grouped = makeSummaryD(f,
                                        HelperFunctions.getUBLASTColumns(),
                                        header=None,
                                        columns_to_plot=['alignment_length',
                                                         'percent_identity',
                                                         'bit_score'],
                                        group_by='target')

        # target_2 is the matching retrovirus without the _ORF_XXX
        # specification - for the plot we just want to know which
        # retrovirus it is
        tab['target_2'] = ["_".join(x.split("_")[:-3]) for x in tab['target']]
        statD[gene] = sD
        tab['gene'] = gene
        bigtab = bigtab.append(tab)

    # plot histograms of alignment length, percent identity and bit_score
    # and a bar chart of the matching target ORF
    makePlots([lengths, sim, score, target],
              ['alignment_length', 'percent_identity', 'bit_score',
               'target_2'],
              ['Length', 'Percent Identity to Target', 'UBLAST Bit Score',
               'Match'],
              ["UBLAST Alignment Lengths of Putative ERV Regions",
               "UBLAST % Identity of Putative ERV Regions",
               "UBLAST Bit Score of Putative ERV Regions",
               "UBLAST Match"],
              ['hist', 'hist', 'hist', 'bar'],
              bigtab,
              genes,
              plotparams,
              outfiles[1:5])
    # plot the counts per gene
    makeGenePlot(tot, bigtab, genes, "Number of ERV Regions per Gene - UBLAST",
                 plotparams, outfiles[5])
    # convert the combined table into a dataframe and save it
    df = pd.DataFrame(statD)
    df = roundAll(df)
    df.to_csv(outfiles[0], sep="\t")


def FastaLengths(F):
    '''
    Returns all the sequence lengths in a dictionary based on a Fasta file
    '''
    return (len(x) for x in F.values())


def summariseORFs(infiles, outfiles, log, genes, plotparams):
    '''
    Summarise output of ORF detection.
    Run on the output from the getORFs function.
    '''
    # generate three empty plots
    dpi = int(plotparams['dpi'])
    tot, lengths, strands = makeInitialFigures(3, dpi=dpi)
    log.info("Plotting summary information for ORFs")

    # make a dictionary for the summary stats
    statD = dict()
    bigtab = pd.DataFrame()
    for i, gene in enumerate(genes):
        f = getInfiles(infiles, gene, 2)

        # retrieve the summary stats for length and length per chromosome
        sD, tab, grouped = makeSummaryD(f,
                                        HelperFunctions.getUBLASTColumns(),
                                        header=None,
                                        columns_to_plot=['length'],
                                        group_by='chrom', maketab=True)
        statD[gene] = sD
        tab['gene'] = gene
        bigtab = bigtab.append(tab)
    # plot a histogram of ORF lengths for each gene and a bar chart of ORF
    # strands
    makePlots([lengths, strands],
              ['length', 'strand'],
              ['Length', 'strand'],
              ["ORF Length",
               "ORF Strand"],
              ['hist', 'bar'],
              bigtab,
              genes,
              plotparams,
              outfiles[1:3])
    # plot the counts per gene
    makeGenePlot(tot, bigtab, genes, "Number of ORFs per Gene",
                 plotparams, outfiles[3])
    # make and save a dataframe of the summary stats
    df = pd.DataFrame(statD)
    df = roundAll(df)
    df.to_csv(outfiles[0], sep="\t")


def summariseGroups(infiles, outfiles, log, genes, plotparams):
    '''
    Summarise output of classifying ORFs into groups based on sequence
    similarity to a known retrovirus.
    Run on the output from the assignGroups function.
    '''
    # make four plots
    dpi = int(plotparams['dpi'])
    tot, lengths, genus, groups = makeInitialFigures(
        4, dpi=dpi, by=[4, 4, 4, 8])
    log.info("Plotting summary information for Screen section")
    statD = dict()
    bigtab = pd.DataFrame()

    for i, gene in enumerate(genes):
        # find the input files for this gene
        f = getInfiles(infiles, gene, 0)
        
        # calculate the summary stats for ORF length and ORF length per genus
        sD, tab, grouped = makeSummaryD(f,
                                        None,
                                        header=0,
                                        columns_to_plot=['length'],
                                        group_by='genus')
        statD[gene] = sD
        tab['gene'] = gene
        bigtab = bigtab.append(tab)
    bigtab = bigtab.sort_values(['gene', 'chrom', 'start'])
    bigtab.to_csv(outfiles[0], sep="\t", index=None)
    # plot a histogram of sequence lengths and bar charts of counts per
    # genus and per group. Group is the small retrovirus group of the
    # most similar reference sequence
    makePlots([lengths, genus, groups],
              ['length', 'genus', 'group'],
              ['Length', 'Genus', 'Group'],
              ["Length",
               "Genus",
               "Group"],
              ['hist', 'bar', 'bar'],
              bigtab,
              genes,
              plotparams,
              outfiles[1:4])
    # plot the counts per gene
    makeGenePlot(tot, bigtab, genes, "Number of ERV Regions per Gene",
                 plotparams, outfiles[4])


def summariseClassify(in_fastas, in_trees, outfiles, genes,
                      plotparams, log):
    '''
    Summarises the output of the classify functions.
    Generates a table with the number of ORFs collapsed into each monophyletic
    group of newly identified retroviruses.
    '''
    log.info("Plotting summary information for groups")
    # There is a fasta file in this directory for each group with all the ORFs
    # in the group - count the sequences in these to get the group size.
    groups = os.listdir("group_lists.dir")
    rows = []
    for fasta in in_fastas:
        # get the basename of the fasta file
        stem = os.path.basename(fasta).split(".")[-2]
        # get the gene and genus
        gene, genus = stem.split("_")
        for group in groups:
            # find the right fasta file
            if stem in group and group.endswith(".fasta"):
                groupnam = group.split(".")[0]
                # count how many sequences are in it
                F = Fasta.makeFastaDict("group_lists.dir/%s" % group)
                count = len(F)
                # make a row for the table
                rows.append([gene, genus, groupnam, count])
    # convert the results into a dataframe and save it
    results = pd.DataFrame(rows, columns=['gene', 'genus', 'group', 'count'])
    results.to_csv(outfiles[0], sep="\t", index=None)
    # plot this information in a bar chart for each gene
    makeClassifyPlots(results, genes, plotparams, outfiles[1])


def sortkey(x):
    '''
    Key to sort strings by number where number is the last item in a string
    delimted by "_" (e.g. x_x_x_1).
    '''
    return (x.str.split("_").str.get(-1).astype(int))


def makeClassifyPlots(tab, genes, plotparams, outfile):
    '''
    Generates a bar chart for each gene showing the number of ORFs collapsed
    into each monophyletic group of newly identified ORFs, organised by genus.
    '''
    # the plot needs to be wider if there are more groups
    f = plt.figure(figsize=(len(tab) / 10, 12))
    for i, gene in enumerate(genes):
        # get the rows in the table for this gene
        subtab = tab[tab['gene'] == gene]
        sp = f.add_subplot(3, 1, (i+1))
        x = 0

        # this is to keep track of the bar names, where to draw the lines
        # between the different genera and how tall each bar was so the
        # axis limits can be set
        splits = []
        counts = []
        ticks = []

        # plot one genus at a time
        for genus in set(tab['genus']):
            # isolate this genus
            genustab = subtab[subtab['genus'] == genus]
            # sort by group name index
            genustab = genustab.sort_values('group',
                                            key=lambda x: sortkey(x))
            # draw the bar chart
            sp.bar(np.arange(x, x+len(genustab)), genustab['count'],
                   color=plotparams['%s_colour' % gene])
            # keep track of bar heights
            counts += list(genustab['count'])
            ticks += list(genustab['group'])
            x += len(genustab)
            # draw a line after this genus
            splits.append(x - 0.5)
        # draw vertical lines between genera
        sp.vlines(splits, 0, (max(counts) * 1.1))

        # tidy up the axes
        sp.set_ylim(0, max(counts) * 1.1)
        sp.set_xlim(0, len(subtab))
        sp.set_xticks(np.arange(len(subtab)))
        sp.set_xticklabels(ticks, rotation='vertical')
        sp.set_xlabel("Group")
        sp.set_ylabel("Number of ORFs")

        # add the title and the total number of ORFs
        sp.set_title("%s: %s ORFs" % (gene,  sum(subtab['count'])))
    # arrange everything on the page
    f.tight_layout()
    # save and close
    dpi = int(plotparams['dpi'])
    f.savefig(outfile, dpi=dpi,
              bbox_inches='tight')
    plt.close()


def summariseERVRegions(infiles, outfiles, genes, plotparams, log):
    '''
    Plots a bar chart of the total number of ERV regions with each combination
    of retroviral genes
    '''
    # Make the empty plot
    dpi = int(plotparams['dpi'])
    f = plt.figure(figsize=(6, 6), dpi=dpi)
    a = f.add_subplot(111)

    # Read the ERV region table
    tab = pd.read_csv(infiles[0], sep="\t")

    # Get the prefix showing the genes in the region
    tab['typ'] = ["_".join(x.split("_")[:-1]) for x in tab['name']]

    # sort by the type of region
    tab = tab.sort_values('typ')

    # count the number of each type
    typ_counts = tab.groupby('typ').size()

    # draw a bar chart of the counts
    a.bar(np.arange(len(typ_counts)), typ_counts,
          color=plotparams['other_colour'])

    # tidy up the axes
    a.set_xticklabels(list(typ_counts.index), rotation='vertical')
    a.set_xticks(np.arange(len(typ_counts)))
    a.set_ylabel("Number of Regions")
    a.set_xlabel("ORFs Present")

    # add the title
    a.set_title("Multi-Gene Regions")

    # save and close
    dpi = int(plotparams['dpi'])
    f.savefig(outfiles[1], dpi=dpi, bbox_inches='tight')
