# AUTOGENERATED! DO NOT EDIT! File to edit: 01_fuzzifiers.ipynb (unless otherwise specified).

__all__ = ['Fuzzifier', 'CrispFuzzifier', 'LinearFuzzifier', 'ExponentialFuzzifier',
           'QuantileConstantPiecewiseFuzzifier', 'QuantileLinearPiecewiseFuzzifier']

# Cell

import numpy as np
import matplotlib.pyplot as plt
import pytest
from scipy.optimize import curve_fit

# Cell

class Fuzzifier:
    def __init__(self, xs=None, mus=None):
        self.xs = xs
        self.mus = mus

    def get_r_to_mu(self,
                    sq_radius, # was SV_square_distance
                    x_to_sq_dist): # was estimated_square_distance_from_center
        '''Transforms the square distance between center of the learnt sphere
        and the image of a point in original space into the membership degree
        of the latter to the induced fuzzy set.

        Not implemented in the base fuzzifier class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        raise NotImplementedError(
        'the base class does not implement get_r_to_mu method')

    def get_membership(self,
                       sq_radius, # was SV_square_distance
                       x_to_sq_dist, # was estimated_square_distance_from_center
                       return_profile=False):
        '''Return the induced membership function.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).

        - `return_profile` flag triggering the generation of the graph
          of the fuzzifier to be returned alongside the fuzzifier itself
          (bool, default=False).

        Returns a list containing

        - the membership function if `return_profile` is `False`, and
        - the membership function and the salient coordinates of the graph of
          the fuzzifier if `return_profile` is `True`.
        '''
        r_to_mu = self.get_r_to_mu(sq_radius, x_to_sq_dist)

        def estimated_membership(x):
            r = x_to_sq_dist(np.array(x))
            return r_to_mu(r)

        result = [estimated_membership]

        if return_profile:
            rdata = list(map(x_to_sq_dist, self.xs))
            rdata_synth = np.linspace(0, max(rdata)*1.1, 200)
            estimate = list(map(r_to_mu, rdata_synth))
            result.append([rdata, rdata_synth, estimate])

        return result

    def __str__(self):
        return self.__repr__()

    def __eq__(self, other):
        return type(self) == type(other)

    def __ne__(self, other):
        return not self == other

    def __hash__(self):
        return hash(self.__repr__())

    def __nonzero__(self):
        return True

# Cell

class CrispFuzzifier(Fuzzifier):
    def __init__(self, xs=None, mus=None, profile='fixed'):
        r'''Creates an instance of `CrispFuzzifier`

        - `xs`: objects in training set (iterable).

        - `mus`: degree memberships in training set (iterable).

        - `profile`: method to be used in order to fix the fuzzifier
          threshold ('fixed' used the sphere radius, 'infer' fits using
          data in `self.xs` and `self.mus`).
        '''

        super().__init__(xs, mus)
        self.profile = profile
        if profile not in ['fixed', 'infer']:
            raise ValueError(f"'profile' parameter should be equal to "
                             "'fixed' or 'infer' (provided value: {profile})")

        if profile == 'infer' and (xs is None or mus is None):
            raise ValueError(f"all arguments to the constructor of "
                             "CrispFuzzifier should be specified when "
                             "profile='infer'")

        self.name = 'Crisp'
        self.latex_name = '$\\hat\\mu_{\\text{crisp}}$'

    def get_r_to_mu(self, sq_radius, x_to_sq_dist):
        r'''Returns a function that transforms the square distance between
        center of the learnt sphere and the image of a point in original
        space into a crisp membership degree having the form

        $$ \mu(r) = \begin{cases}
          1 & \text{if $r \leq r_\text{crisp}$,} \\
          0 & \text{otherwise.}
        \end{cases}$$

        The threshold is set to the learnt square radius of the sphere if the
        `profile` attribute of the class have been set to `fixed`, and
        induced via interpolation of the `xs` and `mus` attributes when it is
        has been set to `infer`.

        **Note** This function is meant to be called internally by the
        `get_membership` method in the base `Fuzzifier` class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        if self.profile == 'fixed':
            return lambda r: 1 if r <= sq_radius else 0

        elif self.profile == 'infer':
            rdata = np.fromiter(map(x_to_sq_dist, self.xs), dtype=float)

            def r_to_mu_prototype(r, r_crisp):
                result = np.ones(len(r))
                result[r > r_crisp] = 0
                return result


            popt, _ = curve_fit(r_to_mu_prototype, rdata, self.mus)
                                # bounds=((0,), (np.inf,)))

            if popt[0] < 0:
                raise ValueError('Profile fitting returned a negative parameter')
            return lambda r: r_to_mu_prototype([r], *popt)[0]
        else:
            raise ('This should never happen. '
                   'Check CrispFuzzifier constructor.')

    def __repr__(self):
        xs_repr = self.xs.__repr__()
        mus_repr = self.mus.__repr__()
        self_repr = f'CrispFuzzifier({xs_repr}, {mus_repr}'
        if self.profile != 'fixed':
            self_repr += f', profile={self.profile}'
        self_repr += ')'
        return self_repr

# Cell

class LinearFuzzifier(Fuzzifier):
    def __init__(self, xs=None, mus=None, profile='fixed'):
        r'''Creates an instance of `LinearFuzzifier`

        - `xs`: objects in training set (iterable).

        - `mus`: degree memberships in training set (iterable).

        - `profile`: method to be used in order to fix the fuzzifier
          threshold ('fixed' used the sphere radius, 'infer' fits using
          data in `self.xs` and `self.mus`).
        '''

        super().__init__(xs, mus)

        if profile not in ['fixed', 'infer']:
            raise ValueError(f"'profile' parameter should be equal to "
                             "'fixed' or 'infer' (provided value: {profile})")

        if profile == 'infer' and (xs is None or mus is None):
            raise ValueError(f"all arguments to the constructor of "
                             "LinearFuzzifier should be specified when "
                             "profile='infer'")

        self.profile = profile

        self.name = 'Linear'
        self.latex_name = '$\\hat\\mu_{\\text{lin}}$'


    def get_r_to_mu(self,
                    sq_radius, # was SV_square_distance
                    x_to_sq_dist): #was estimated_square_distance_from_center
        r'''Returns a function that transforms the square distance between
        center of the learnt sphere and the image of a point in original
        space into a membership degree having the form

        $$ \mu(r) = \begin{cases}
          1    & \text{if $r \leq r_\text{crisp}$,} \\
          l(r) & \text{otherwise,}
        \end{cases}$$

        where $l$ is a linear function decreasing from 1 to 0. The slope of
        this linear function is chosen so that the latter contains the point
        (sq_radius, 0.5) if the `profile` attribute of the class
        have been set to `fixed`, and induced via interpolation of the `xs`
        and `mus` attributes when it is has been set to `infer`.

        **Note** This function is meant to be called internally by the
        `get_membership` method in the base `Fuzzifier` class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        rdata = np.fromiter(map(x_to_sq_dist, self.xs), dtype=float)

        r_1_guess = np.median([x_to_sq_dist(x)
                               for x, mu in zip(self.xs, self.mus)
                               if mu>=0.99])

        if self.profile == 'fixed':
            def r_to_mu_prototype(r, r_1):
                return [np.clip(1 - 0.5 * (rr-r_1) / (sq_radius - r_1), 0, 1)
                        for rr in r]

            popt, _ = curve_fit(r_to_mu_prototype,
                                rdata, self.mus,
                                p0=(r_1_guess,),
                                bounds=((0,), (np.inf,)))

        elif self.profile == 'infer':

            def r_to_mu_prototype(r, r_1, r_0):
                return [np.clip(1 - (r_1 - rr) / (r_1 - r_0), 0, 1)
                        for rr in r]

            popt, _ = curve_fit(r_to_mu_prototype,
                                rdata, self.mus,
                                p0=(r_1_guess, 10*sq_radius),
                                bounds=((0, 0), (np.inf, np.inf,)))
        else:
            raise ValueError('This should never happen.'
                             ' Check LinearFuzzifier constructor.')
        if min(popt) < 0:
            raise ValueError('Profile fitting returned a negative parameter')

        return lambda r: r_to_mu_prototype([r], *popt)[0]


    def __repr__(self):
        xs_repr = self.xs.__repr__()
        mus_repr = self.mus.__repr__()
        self_repr = f'LinearFuzzifier({xs_repr}, {mus_repr}'
        if self.profile != 'fixed':
            self_repr += f', profile={self.profile}'
        self_repr += ')'
        return self_repr


# Cell
class ExponentialFuzzifier(Fuzzifier):
    def __init__(self, xs=None, mus=None,
                 profile='fixed', alpha=None, sample=None):
        r'''Creates an instance of `ExponentialFuzzifier`

        - `xs`: objects in training set (iterable).

        - `mus`: degree memberships in training set (iterable).

        - `profile`: method to be used in order to fix the exponential
          decay ('fixed' used the sphere radius, 'infer' fits using
          data in `self.xs` and `self.mus`, and `alpha` relies on a
          manually set `alpha` parameter).

        - `alpha`: decaying degree of the exponential (float in (0, 1)).
        '''

        super().__init__(xs, mus)

        if profile not in ['infer', 'fixed', 'alpha']:
            raise ValueError(f"'profile' parameter should be equal to "
                             "'infer', 'fixed' or 'alpha' "
                             "(provided value: {profile})")

        if profile == 'infer' and (xs is None or mus is None):
            raise ValueError(f"arguments xs and mus to the constructor of "
                             "ExponentialFuzzifier should be specified when "
                             "profile='infer'")

        if alpha is not None and profile != 'alpha':
            raise ValueError(f"'alpha' value is specified, but 'profile' "
                             f"is set to '{profile}'")

        if profile == 'alpha':
            if alpha is None:
                raise ValueError("alpha must be set to a float when"
                                 "'profile' is 'alpha'")
            if alpha <= 0 or alpha >= 1:
                raise ValueError("alpha must belong to (0, 1)"
                                 "'profile' is 'alpha'")
            self.name = f'Exponential({alpha})'
            self.latex_name = \
                      r'$\hat\mu_{{\text{{exp}},{:.3f}}}$'.format(alpha)

            self.alpha = alpha
        else:
            self.name = 'Exponential'
            self.latex_name = r'$\hat\mu_{\text{nexp}}$'

        self.profile = profile

        self.sample = sample


    def get_r_to_mu(self,
                    sq_radius, # was SV_square_distance
                    x_to_sq_dist): #was estimated_square_distance_from_center
        r'''Returns a function that transforms the square distance between
        center of the learnt sphere and the image of a point in original
        space into a membership degree having the form

        $$ \mu(r) = \begin{cases}
          1    & \text{if $r \leq r_\text{crisp}$,} \\
          e(r) & \text{otherwise,}
        \end{cases}$$

        where $e$ is an exponential function decreasing from 1 to 0. The
        shape of this function is chosen so that the latter contains the
        point (sq_radius, 0.5) if the `profile` attribute of the class
        have been set to `fixed`, induced via interpolation of the `xs`
        and `mus` attributes when it is has been set to `infer`, and
        manually set via the `alpha` parameter of the constructor when
        `profile='alpha'`.

        **Note** This function is meant to be called internally by the
        `get_membership` method in the base `Fuzzifier` class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        r_1_guess = np.median([x_to_sq_dist(x)
                               for x, mu in zip(self.xs, self.mus)
                               if mu>=0.99])

        s_guess = (sq_radius - r_1_guess) / np.log(2)

        rdata = np.fromiter(map(x_to_sq_dist, self.xs), dtype=float)

        if self.profile == 'fixed':
            def r_to_mu_prototype(r, r_1):
                return [np.clip(np.exp(-(rr-r_1)/(sq_radius-r_1) * np.log(2)),
                                0, 1) for rr in r]

            popt, _ = curve_fit(r_to_mu_prototype, rdata, self.mus,
                                p0=(r_1_guess,))
                                #bounds=((0,), (np.inf,)))
            return lambda r: r_to_mu_prototype([r], *popt)[0]

        elif self.profile == 'infer':
            def r_to_mu_prototype(r, r_1, s):
                return [np.clip(np.exp(-(rr - r_1) / s), 0, 1) for rr in r]

            popt, _ = curve_fit(r_to_mu_prototype, rdata, self.mus,
                                #p0=(r_1_guess, s_guess),
                                bounds=((0, 0), (np.inf, np.inf)))

            return lambda r: r_to_mu_prototype([r], *popt)[0]

        elif self.profile == 'alpha':
            r_sample = map(x_to_sq_dist, self.sample \
                            if self.sample is not None else self.xs)

            q = np.percentile([s - sq_radius for s in r_sample
                                             if s > sq_radius],
                              100*self.alpha)

            return lambda r: np.clip(np.exp(np.log(self.alpha)/q * (r - sq_radius)),
                           0, 1)
        else:
            raise ValueError('This should not happen. Check the constructor')

    def __repr__(self):
        xs_repr = self.xs.__repr__()
        mus_repr = self.mus.__repr__()
        self_repr = f'ExponentialFuzzifier({xs_repr}, {mus_repr}'
        if self.profile != 'fixed':
            self_repr += f', profile={self.profile}'
        if self.alpha is not None:
            self_repr += f', alpha={self.alpha}'
        if self.sample is not None:
            self_repr += f', sample={self.sample.__repr__()}'
        self_repr += ')'
        return self_repr


# Cell

class QuantileConstantPiecewiseFuzzifier(Fuzzifier):
    def __init__(self, xs=None, mus=None, sample=None):
        r'''Creates an instance of `QuantileConstantPiecewiseFuzzifier`

        - `xs`: objects in training set (iterable).

        - `mus`: degree memberships in training set (iterable).

        - `sample`: sample to be used in order to build quantile instead of
          training set.
        '''

        super().__init__(xs, mus)

        if sample is None and xs is None:
            raise ValueError(f"either the xs or sample argument of the "
                             "constructor of "
                             "QuantileConstantPiecewiseFuzzifier should be "
                             "specified")

        self.sample = sample

        self.name = 'QuantileConstPiecewise'
        self.latex_name = '$\\hat\\mu_{\\text{qconst}}$'


    def get_r_to_mu(self,
                    sq_radius, # was SV_square_distance
                    x_to_sq_dist): #was estimated_square_distance_from_center
        r'''Returns a function that transforms the square distance between
        center of the learnt sphere and the image of a point in original
        space into a constant piecewise membership degree whose steps
        are chosen according to the quartiles of square distances between
        images of points and center of the learnt sphere.

        **Note** This function is meant to be called internally by the
        `get_membership` method in the base `Fuzzifier` class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        r_sample = np.fromiter(map(x_to_sq_dist, self.sample \
                      if self.sample is not None else self.xs), dtype=float)
        external_dist = [s-sq_radius
                         for s in r_sample if s > sq_radius]

        if external_dist:
            m = np.median(external_dist)
            q1 = np.percentile(external_dist, 25)
            q3 = np.percentile(external_dist, 75)
        else:
            m = q1 = q3 = 0

        def r_to_mu(r):
            return 1 if r <= sq_radius \
                     else 0.75 if r <= sq_radius + q1 \
                     else 0.5 if r <= sq_radius + m \
                     else 0.25 if r <= sq_radius + q3 \
                     else 0

        return r_to_mu

    def __repr__(self):
        return 'QuantileConstantPiecewiseFuzzifier({}, {})'.format(self.xs, self.mus)
        self_repr = 'QuantileConstantPiecewiseFuzzifier('
        xs_repr = self.xs.__repr__()
        mus_repr = self.mus.__repr__()
        self_repr += f'{xs_repr}, {mus_repr}'
        if self.sample is not None:
            self_repr += f', sample={self.sample.__repr__()}'
        self_repr += ')'
        return self_repr

# Cell

class QuantileLinearPiecewiseFuzzifier(Fuzzifier):
    def __init__(self, xs=None, mus=None, sample=None):
        r'''Creates an instance of `QuantileLinearPiecewiseFuzzifier`

        - `xs`: objects in training set (iterable).

        - `mus`: degree memberships in training set (iterable).

        - `sample`: sample to be used in order to build quantile instead of
          training set.
        '''

        super().__init__(xs, mus)

        if sample is None and xs is None:
            raise ValueError(f"either the xs or sample argument of the "
                             "constructor of "
                             "QuantileLinearPiecewiseFuzzifier should be "
                             "specified")

        self.sample = sample

        self.name = 'QuantileLinPiecewise'
        self.latex_name = '$\\hat\\mu_{\\text{qlin}}$'


    def get_r_to_mu(self,
                    sq_radius, # was SV_square_distance
                    x_to_sq_dist): #was estimated_square_distance_from_center
        r'''Returns a function that transforms the square distance between
        center of the learnt sphere and the image of a point in original
        space into a linear piecewise membership degree whose steps
        are chosen according to the quartiles of square distances between
        images of points and center of the learnt sphere.

        **Note** This function is meant to be called internally by the
        `get_membership` method in the base `Fuzzifier` class.

        - `sq_radius`: squared radius of the learnt sphere (float).

        - `x_to_sq_dist`: mapping of a point in original space into the
          square distance of its image from the center of the learnt sphere
          (function).
        '''

        r_sample = np.fromiter(map(x_to_sq_dist, self.sample \
                      if self.sample is not None else self.xs), dtype=float)

        external_dist = [s-sq_radius
                         for s in r_sample if s > sq_radius]
        if external_dist:
            m = np.median(external_dist)
            q1 = np.percentile(external_dist, 25)
            q3 = np.percentile(external_dist, 75)
            mx = np.max(r_sample) - sq_radius
        else:
            m = q1 = q3 = mx = 0

        def r_to_mu(r):
            ssd = sq_radius
            return 1 if r <= ssd \
                 else (-r + ssd)/(4*q1) + 1 if r <= ssd + q1 \
                 else (-r + ssd + q1)/(4*(m-q1)) + 3.0/4 if r <= ssd + m \
                 else (-r + ssd + m)/(4*(q3-m)) + 1./2 if r <= ssd + q3 \
                 else (-r + ssd + q3)/(4*(mx-q3)) + 1./4 if r <= ssd+mx \
                 else 0

        return r_to_mu

    def __repr__(self):
        return 'QuantileLinearPiecewiseFuzzifier({}, {})'.format(self.xs, self.mus)
