'''
Created on 2020-09-17 19:10:47
Last modified on 2020-09-30 11:39:22

@author: L. F. Pereira (lfpereira@fe.up.pt))
'''

# imports

# standard library
import os
import pickle
from collections import OrderedDict

# local library
import f3dasm
from ..utils.file_handling import InfoReport
from ..post_processing import collect_raw_data


# object definition

def create_main_file(example_name, doe_variables, points, sim_info,
                     fixed_variables=None, additional_variables=None,
                     additional_info=None, transform_inputs=None,
                     data_filename='DoE.pkl',):
    '''
    Create file where all the information required to run simulations is
    contained.

    Parameters
    ----------
    example_name : str
        Folder name.
    doe_variables: dicts
        Variables of the DoE (generated by design of experiments) with lower
        and upper bounds.
    points: pandas.DataFrame
        Design of experiments points.
    sim_info : dict or list of dict
        All the information required to create each simulation. If sequential
        simulations, then use a list of dicts.
    fixed_variables : dict
        Model input variables that are kept fix during the design of experiments.
    additional_variables: dict
        Model input variables that change in each DoE but are not controlled by
        the design of experiments.
    additional_info : dict
        Information about the problem to store for later use.
    transform_inputs : str
        Interface function to transform inputs from raw to the name required
        by the model.
    data_filename : str
        Name of main file.
    '''

    # manipulate sim_info
    sim_info_dict = OrderedDict()
    if type(sim_info) is list:
        for sim_info_ in sim_info:
            name, sinfo = _manipulate_sim_dict(sim_info_)
            sim_info_dict[name] = sinfo
    else:
        name, sinfo = _manipulate_sim_dict(sim_info)
        sim_info_dict[name] = sinfo

    # create data dictionary with required information
    data = {'doe_variables': doe_variables,
            'points': points,
            'sim_info': sim_info_dict,
            'run_info': {'missing_sims': set(range(len(points))),
                         'running_sims': set(),
                         'error_sims': set(),
                         'successful_sims': set()},
            'version': f3dasm.__version__}

    # add facultative information
    facultative_vars = ['fixed_variables', 'additional_variables',
                        'transform_inputs', 'additional_info']
    for var in facultative_vars:
        if var is not None:
            data[var] = eval(var)

    # create directory and save pkl file
    os.mkdir(example_name)
    with open(os.path.join(example_name, data_filename), 'wb') as file:
        pickle.dump(data, file)


def create_sim_info(name, abstract_model, job_info, post_processing_fnc=None,
                    **kwargs):
    '''
    It shows all the keys that can be used when creating simulation information.

    Parameters
    ----------
    # TODO: finish docstrings
    abstract_model : str
        Objects used to create numerical models.

    post_processing_fnc : str or array of str
        Objects used to post-process numerical models, if `abstract_model` does
        is not a child of `f3dasm.abaqus.modelling.model.BasicModel`

    # TODO: change place
    sim_info : dict
        OrderedDict with all the information required to instantiate a model
        (but that is not a model geometric or material variable, i.e. not in
        `doe_variables`, `fixed_variables` or `additional_variables`).
        `job_info` is mandatory (minimum information required: 'name').

    '''

    # verification of job info
    if 'name' not in job_info.keys():
        raise Exception('job info must contain at least `name`')

    # create dict
    sim_info = {'name': name,
                'abstract_model': abstract_model,
                'job_info': job_info,
                'post_processing_fnc': post_processing_fnc}
    sim_info.update(kwargs)

    return sim_info


def get_updated_sims_state(raw_data=None, example_name=None,
                           points=None, sims_dir_name='analyses',):
    '''
    Parameters
    ----------
    raw_data : pd.Series or str or None.
        Data is gatherer according to `raw_data` type. The possibilities are:
            None: simulation folders
            str: raw data file
            pandas.Series: uses itself.
    points : array
        If None, considers all created simulation folders or folders available
        in `raw_data` file. Only applicable if `raw_data` is a path to a file
        or None.
    '''

    # get raw data
    if type(raw_data) is str or raw_data is None:
        raw_data = collect_raw_data(example_name, sims_dir_name=sims_dir_name,
                                    sim_numbers=points, delete=False,
                                    raw_data_filename=raw_data)

    # getting sims state
    error_sims = []
    successful_sims = []
    for point, data_sim in raw_data.iteritems():
        success = data_sim.get('success', None)
        if success:
            successful_sims.append(point)
        elif success is False:
            error_sims.append(point)

    return set(error_sims), set(successful_sims)


def get_sims_info(example_name, data_filename='DoE.pkl',
                  sims_dir_name='analyses', print_info=True, report=''):
    # TODO: move to stats?

    # initialization
    info = InfoReport(sections=['run_info'])
    run_info_sec = info['run_info']

    # access data
    with open(os.path.join(example_name, data_filename), 'rb') as file:
        data = pickle.load(file)

    # running simulations
    running_sims = data['run_info']['running_sims']
    error_sims_, successful_sims_ = get_updated_sims_state(
        example_name=example_name, points=running_sims,
        sims_dir_name=sims_dir_name, raw_data=None,)
    n_running_sims = len(running_sims)
    n_running_sims_miss = len(set(running_sims) - error_sims_ - successful_sims_)

    # other simulations
    n_missing_sims = len(data['run_info']['missing_sims']) + n_running_sims_miss
    n_error_sims = len(data['run_info']['error_sims']) + len(error_sims_)
    n_successful_sims = len(data['run_info']['successful_sims']) + len(successful_sims_)
    n_run = n_error_sims + n_successful_sims
    n_total = n_missing_sims + n_run

    # compute information
    if n_running_sims:
        run_info_sec.add_info(
            'Missing simulations (running): {}/{} ({:.1f}%)'.format(
                n_running_sims_miss, n_running_sims,
                n_running_sims_miss / n_running_sims * 100))
    run_info_sec.add_info('Missing simulations (total): {}/{} ({:.1f}%)'.format(
        n_missing_sims, n_total, n_missing_sims / n_total * 100))
    if n_run:
        run_info_sec.add_info('With errors: {}/{} ({:.1f}%)'.format(
            n_error_sims, n_run, n_error_sims / n_run * 100))
        run_info_sec.add_info('Successful: {}/{} ({:.1f}%)'.format(
            n_successful_sims, n_run, n_successful_sims / n_run * 100))

    # print information
    if print_info:
        info.print_info(print_headers=False)

    # create report
    if report:
        with open(report, 'w') as file:
            info.write_report(file, print_headers=False)

    return info


def update_run_info(example_name, data_filename='DoE.pkl',
                    sims_dir_name='analyses'):
    '''
    Updates information about simulations. Assumes simulations are not being
    ran. It is supposed to correct possible outdated files due to running of
    simulations in different machines.
    '''
    # TODO: review

    # access data
    with open(os.path.join(example_name, data_filename), 'rb') as file:
        data = pickle.load(file)

    # compute information
    points = list(range(len(data['points'])))
    error_sims, successful_sims = get_updated_sims_state(
        example_name, points, sims_dir_name)
    missing_sims = list(set(points) - set(error_sims) - set(successful_sims))

    # dump information
    data['run_info']['missing_sims'] = missing_sims
    data['run_info']['running_sims'] = []
    data['run_info']['error_sims'] = error_sims
    data['run_info']['successful_sims'] = successful_sims
    with open(os.path.join(example_name, data_filename), 'wb') as file:
        pickle.dump(data, file)


def _manipulate_sim_dict(sim_dict):
    name = sim_dict['name']
    del sim_dict['name']

    return name, sim_dict
