"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        // For some reason path mapping to 'things/{thingName}/shadow/name/{shadowName}' results in 403 error, hence this shortcut
        this.addResourceMethod(namedShadowResource, props, 'topics/$aws/things/{thingName}/shadow/name/{shadowName}/update', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
        if (props.apiGatewayCreateApiKey === true) {
            const cfnMethod = apiMethod.node.findChild('Resource');
            cfnMethod.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W59',
                            reason: 'When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE'
                        }]
                }
            };
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "1.107.0" };
//# sourceMappingURL=data:application/json;base64,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