"""Module for surfer data."""
from typing import Iterable

# isort: split
from ansys.meshing.prime.autogen.surfer import Surfer as _Surfer

# isort: split
from ansys.meshing.prime.autogen.coreobject import CoreObject
from ansys.meshing.prime.autogen.surferstructs import (
    LocalSurferParams,
    LocalSurferResults,
    SurferParams,
    SurferResults,
)
from ansys.meshing.prime.core.model import Model


class Surfer(CoreObject):
    """Provides surfer operations.

    Parameters
    ----------
    model : Model
        Model to operate in.
    """

    __doc__ = _Surfer.__doc__

    def __init__(self, model: Model):
        """Initialize a surfer instance."""
        self._model = model

    def mesh_topo_faces(
        self, part_id: int, topo_faces: Iterable[int], params: SurferParams
    ) -> SurferResults:
        """Perform meshing on TopoFaces with given parameters.

        Parameters
        ----------
        part_id : int
            ID of part.
        topo_faces : Iterable[int]
            IDs of the TopoFaces.
        params : SurferParams
            Surfer parameters.

        Returns
        -------
        SurferResults
            Results from performing meshing on the TopoFaces.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = prime.SurferParams(model)
        >>> results = surfer.mesh_topo_faces(part_id, topo_faces, surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.mesh_topo_faces(topo_faces, params)

    def remesh_face_zonelets_locally(
        self,
        part_id: int,
        face_zonelets: Iterable[int],
        register_id: int,
        local_surfer_params: LocalSurferParams,
    ) -> LocalSurferResults:
        """Remesh face zonelets locally at the registered faces with given parameters.

        Parameters
        ----------
        part_id : int
            ID of part.
        face_zonelets : Iterable[int]
            IDs of the face zonelets.
        register_id : int
            Register ID of the target faces.
        local_surfer_params : LocalSurferParams
            Local surfer parameters.

        Returns
        -------
        LocalSurferResults
            Results from remeshing the face zonelets.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> local_surfer_params = prime.LocalSurferParams(model)
        >>> results = surfer.remesh_face_zonelets_locally(part_id,
                        face_zonelets, register_id, local_surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.remesh_face_zonelets_locally(
                face_zonelets, register_id, local_surfer_params
            )

    def remesh_face_zonelets(
        self,
        part_id: int,
        face_zonelets: Iterable[int],
        edge_zonelets: Iterable[int],
        params: SurferParams,
    ) -> SurferResults:
        """Perform meshing on face zonelets with given parameters.

        Parameters
        ----------
        part_id : int
            ID of part.
        face_zonelets : Iterable[int]
            IDs of the face zonelets.
        edge_zonelets : Iterable[int]
            IDs of the edge zonelets.
        params : SurferParams
            Surfer parameters.

        Returns
        -------
        SurferResults
            Results from meshing the face zonelets.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = prime.SurferParams(model)
        >>> results = surfer.remesh_face_zonelets(part_id,
                        face_zonelets, edge_zonelets, surfer_params)

        """
        with _Surfer(model=self._model, part_id=part_id) as surfer:
            return surfer.remesh_face_zonelets(face_zonelets, edge_zonelets, params)

    def initialize_surfer_params_for_wrapper(self) -> SurferParams:
        """Initialize surfer parameters to mesh surfaces generated by the wrapper.

        Returns
        -------
        SurferParams
            Surfer parameters initialized for wrapper inputs.

        Examples
        --------
        >>> surfer = prime.Surfer(model)
        >>> surfer_params = surfer.initialize_surfer_params_for_wrapper()

        """
        with _Surfer(model=self._model, part_id=-1) as surfer:
            return surfer.initialize_surfer_params_for_wrapper()

    def __enter__(self):
        """Run initializing context manager."""
        return self

    def __exit__(self, type, value, traceback):
        """Run when closing the context manager."""
        pass
