# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['emailee']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'emailee',
    'version': '1.0.0',
    'description': 'A robust smtp email send tool, with asyncronous send capability',
    'long_description': '# emailee\n\n[![PyPI Version][pypi-image]][pypi-url]\n[![Build Status][build-image]][build-url]\n[![Code Coverage][coverage-image]][coverage-url]\n[![Code Quality][quality-image]][quality-url]\n[![Code Factor][codefactor-image]][codefactor-url]\n[![versions][versions-image]][versions-url]\n[![license][license-image]][license-url]\n\n<!-- Badges -->\n\n[pypi-image]: https://img.shields.io/pypi/v/emailee\n[pypi-url]: https://pypi.org/project/emailee/\n\n[build-image]: https://github.com/mattwalshdev/emailee/actions/workflows/build.yml/badge.svg\n[build-url]: https://github.com/mattwalshdev/emailee/actions/workflows/build.yml\n\n[coverage-image]: https://codecov.io/gh/mattwalshdev/emailee/branch/main/graph/badge.svg\n[coverage-url]: https://codecov.io/gh/mattwalshdev/emailee\n\n[quality-image]: https://api.codeclimate.com/v1/badges/0360d77f65da3d985d03/maintainability\n[quality-url]: https://codeclimate.com/github/mattwalshdev/emailee\n\n[codefactor-image]: https://www.codefactor.io/repository/github/mattwalshdev/emailee/badge\n[codefactor-url]: https://www.codefactor.io/repository/github/mattwalshdev/emailee\n\n[versions-image]: https://img.shields.io/pypi/pyversions/emailee.svg\n[versions-url]: https://github.com/mattwalshdev/emailee\n\n[license-image]: https://img.shields.io/github/license/mattwalshdev/emailee.svg\n[license-url]: https://github.com/mattwalshdev/emailee/blob/main/LICENSE\n\nA robust SMTP mailer library, with async mail send features built-in using Threads and Multiprocessing. Zero PyPI (production) dependencies.\n\n## Installation\n\nInstall with `pip install emailee`.\n\n## Some quick examples\n\nSend a simple text email\n\n```Python\nimport emailee\n\nmy_email = emailee.Emailee()\nmy_email.sender("john.smith@fakeemail.com")\nmy_email.subject("Test email")\nmy_email.msgContent("some poorly formatted raw text")\nmy_email.sendTo(["jill.smith@fakeemail.com"])\nmy_email.server("smtp.fakeemail.com")\n\nif my_email.ready():\n    my_email.send()\n```\n\nSend multiple asyncronous emails\n\n```Python\nimport emailee\n\nemails_list = [{...}]\nserver_dict = {...}\n\nemails = emailee.AsyncThreads(emails_list, server_dict, outputFile=\'output.txt\')\n```\n\n## Documentation\n\nThere are only three classes exposed to the user of this library, `Emailee`, `AsyncThreads` and `AsyncMP`. The latter two utilise the Emailee class to send emails via the Threading or Multiprocessing async APIs respectively.\n\n### Emailee class\n\nIf you have used the `smtplib` and `email` libraries in the past, you may of found how difficult it is getting the email logic set up properly to handle attachments and different sending scenarios. All customisations are built into Emailee, which takes the guess work out of how to configure.\n\nThe following methods are exposed to the user:\n\n#### Emailee.sender(sender: str, replyTo: str = "") -> None\n\n* **sender** - email address of who is sending the email\n* **replyTo** (optional) - alternative email address the receiver can reply to\n\n#### Emailee.subject(subject: str = "") -> None\n\n* **subject** (optional) - email subject line, maximum of 255 characters\n\n#### Emailee.msgContent(msgText: str = "", msgHTML: str = "") -> None\n\n* **msgText** (optional) - email message in raw text format\n* **msgHTML** (optional) - email message in email HTML format\n\nIt\'s a good idea to use both of these to replicate your email, so users who can\'t read HTML emails can read your fallback raw text email. Look online for HTML email generators\n\n#### Emailee.sendTo(to: List[str] = [], cc: List[str] = [], bcc: List[str] = [], ignoreErrors: bool = False) -> None\n\n* **to** (optional) - list of to addresses to send emails to\n* **cc** (optional) - list of cc addresses to send emails to\n* **bcc** (optional) - list of bcc addresses to send emails to\n* **ignoreErrors** (optional) - all addresses are validated by regex to be in a valid email format. By default, if an email address is found to be invalid the program will raise an exception. If changed to True, the program will ignore and remove invalid emails and continue with sending the email.\n\n#### Emailee.attachmentFiles(attachmentFiles: List[str] = []) -> None\n\n* **attachmentFiles** (optional) - list of attachments for email, can be listed by relative or full path\n\n#### Emailee.server(smtpServer: str, port: int = 0, SSLTLS: str = "", authUsername: str = "", authPassword: str = "", timeout: int = 30) -> None\n\n* **smtpServer** - server name or IP address of SMTP server\n* **port** (optional) - port number for SMTP server, if connection not authenticated and port left blank, port will default to 25. If connection set to SSL or TLS and port left blank, port will default to 465 or 587 respectively.\n* **SSLTLS** (optional) - encrypted connection setting, value either "SSL" or "TLS"\n* **authUsername** (optional) - username for logging into the SMTP server, if authentication required. If left blank, but authPassword filled in, this will default to the sender email.\n* **authPassword** (optional) - password for logging into the SMTP server\n* **timeout** (optional) - server connection and email send timeout limit. Extend this value if on a slow network, or sending emails with large attachments\n\n#### Emailee.ready() -> bool\n\nReturns True if the minimal fields to send an email have been set.\n\n#### Email.send(outputFile: str) -> bool\n\n* **outputFile** (optional) - relative or full path location of a file to write report output metadata to, required for async classes when sending mail, file must be empty or not currently exist\n\nReturns True if successful, does not guarantee email was delivered, just sent.\n\n#### Email.__repr__() -> str\n\nPrint class function, will print a dictionary of email metadata, useful for testing and debugging.\n\n---\n\n### Data for async classes\n\nBoth async classes are very similar in their function and both require emails and server config in the exact same format:\n\n```Python\nmail_list = [\n    {\n        \'sender\': sender # see Emailee.sender()\n        \'replyTo\': replyTo # see Emailee.sender()\n        \'subject\': email_subject # see Emailee.subject()\n        \'msgText\': raw_text_message # see Emailee.msgContent()\n        \'msgHTML\': HTML_formatted_message # see Emailee.msgContent()\n        \'to\': list_of_to_emails # see Emailee.sendTo()\n        \'cc\': list_of_cc_emails # see Emailee.sendTo()\n        \'bcc\': list_of_bcc_emails # see Emailee.sendTo()\n        \'ignoreErrors\': True # see Emailee.sendTo()\n        \'attachmentFiles\': list_of_attachment_file_paths # see Emailee.attachmentFiles()\n    },\n    {...},\n]\n```\n\n```Python\nserver_dict = {\n    \'smtpServer\': SMTP_server # see Emailee.server()\n    \'port\': port_number # see Emailee.server()\n    \'SSLTLS\': \'TLS\' # see Emailee.server()\n    \'authUsername\': authenticated_username # see Emailee.server()\n    \'authPassword\': authenticated_password # see Emailee.server()\n}\n```\n\n### AsyncThreads class\n\nAsyncThreads uses the threading API to enable asyncronous sending of email. Threading only utilises the same CPU core that the Python program is currently running on, so it can only maximise a single core usage, but the benefits are you can throttle how many threads are allowed to run concurrently and it has a lower overhead compared to `AsyncMP`.\n\n#### AsyncThreads(mailList: List[Dict], serverDict: Dict, outputFile: str, maxThreads: int = 10, waitTime: float = 0)\n\n* **mailList** - see mail_list above for format\n* **serverDict** - see server_dict above for format\n* **outputFile** - relative or full path location of a file to write report output metadata to, file must be empty or not currently exist. Mandatory for async classes to handle errors.\n* **maxThreads** - maximum number of concurrent threads to send emails on\n* **waitTime** - wait time in seconds to pause between each thread being spawned, this helps if your SMTP server throttles your connection if you send too many emails in a short time span\n\n### AsyncMP class\n\nAsyncMP uses the multiprocessing API to enable asyncronous sending of email. Multiprocessing utilises all CPU cores of the system the Python program is running on, but does not allow throttling, so you run the risk of maxing out your CPU and RAM, slowing down your system.\n\n#### AsyncMP(mailList: List[Dict], serverDict: Dict, outputFile: str, waitTime: float = 0)\n\n* **mailList** - see mail_list above for format\n* **serverDict** - see server_dict above for format\n* **outputFile** - relative or full path location of a file to write report output metadata to, file must be empty or not currently exist. Mandatory for async classes to handle errors.\n* **waitTime** - wait time in seconds to pause between each process being spawned, this helps if your SMTP server throttles your connection if you send too many emails in a short time span\n\n### Reporting on async output\n\nUpon completion of either async class, you can call the `emailReport()` method to return a metadata list of all emails sent.\nThis won\'t work if an exception is thrown during the sending process, hence the **outputFile** requirement.\nThe output file will be written to after each successful email send so you can analyse against `emailReport()` or see who received emails before a thrown exception or manual cancellation of the sending process.\n\n## Development and Testing\n\nEmailee is built with [**poetry**](https://python-poetry.org/), tested with [**pytest**](https://pytest.org), [**tox**](https://tox.readthedocs.io/en/latest/) and [**coverage**](https://coverage.readthedocs.io/en/coverage-5.5/), type checked with [**mypy**](http://mypy-lang.org/) and formatted with [**black**](https://github.com/psf/black).\n\n### Dev Installation\n\n#### Install with Poetry\n\nClone the repo, `cd` into it and run `poetry install`\n\n#### Install with pip\n\nClone the repo, `cd` into it, build and run a new virtual environment, then open the `pyproject.toml` file and `pip install` all packages listed under **[tool.poetry.dev-dependencies]**\n\n### Testing\n\nTo successfully run local tests you will need to rename `tests/example.test.env.toml` to `tests/test.env.toml` and modify the config inside the file to contain valid email and SMTP server connection data.\n\nTesting and coverage can then be run with `pytest --cov-report term-missing --cov=emailee tests`\n\nTox can be used to run pytest against all support Python environments. Open `tox.ini` and check if you have all versions of Python installed listed under *envlist*.\n\n### Pre-commit hooks\n\nRun `pre-commit install` to install the pre-commit hooks in the `.pre-commit-config.yaml` file. Then run `pre-commit run --all-files` to auto-check every file for issues.\n\nFeel free to send me any changes or feedback.\n\n---\n\nThanks for reading :)\n\n:envelope: :envelope: :envelope: :envelope: :envelope:\n',
    'author': 'Matt Walsh',
    'author_email': 'git@mattwalsh.dev',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'http://github.com/mattwalshdev/emailee',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
