"""
Tests for NEXIUM routing system.
Author: Juste Elysée MALANDILA
"""

import pytest
from nexium.routing import Router, get, post, put, delete


def test_router_creation():
    """Test creating a router."""
    router = Router()
    assert router.prefix == ""
    assert router.routes == []


def test_router_with_prefix():
    """Test router with prefix."""
    router = Router(prefix="/api")
    assert router.prefix == "/api"


def test_add_route():
    """Test adding a route."""
    router = Router()
    
    async def handler():
        return {}
    
    router.add_route("/test", handler, methods=["GET"])
    
    assert len(router.routes) == 1
    assert router.routes[0].path == "/test"


def test_get_decorator():
    """Test GET decorator."""
    router = Router()
    
    @router.get("/users")
    async def list_users():
        return []
    
    assert len(router.routes) == 1
    assert "GET" in router.routes[0].methods


def test_post_decorator():
    """Test POST decorator."""
    router = Router()
    
    @router.post("/users")
    async def create_user():
        return {}
    
    assert len(router.routes) == 1
    assert "POST" in router.routes[0].methods


def test_put_decorator():
    """Test PUT decorator."""
    router = Router()
    
    @router.put("/users/{id}")
    async def update_user(id: int):
        return {}
    
    assert len(router.routes) == 1
    assert "PUT" in router.routes[0].methods


def test_delete_decorator():
    """Test DELETE decorator."""
    router = Router()
    
    @router.delete("/users/{id}")
    async def delete_user(id: int):
        return {}
    
    assert len(router.routes) == 1
    assert "DELETE" in router.routes[0].methods


def test_router_tags():
    """Test router tags."""
    router = Router(tags=["users"])
    
    @router.get("/list")
    async def list_users():
        return []
    
    assert "users" in router.routes[0].tags


def test_include_router():
    """Test including a router."""
    main_router = Router()
    sub_router = Router()
    
    @sub_router.get("/test")
    async def test():
        return {}
    
    main_router.include_router(sub_router, prefix="/sub")
    
    assert len(main_router.routes) == 1
    assert main_router.routes[0].path == "/sub/test"


def test_standalone_decorators():
    """Test standalone decorators."""
    
    @get("/test")
    async def test_get():
        return {}
    
    assert hasattr(test_get, "_route_info")
    assert test_get._route_info["path"] == "/test"
    assert test_get._route_info["methods"] == ["GET"]
